/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    ef.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <algorithm/ef.h>

/* Internal functions */
static void operationa (MonoGraph *g);
static void operationb (MonoGraph *g);
static ChromonList *operationc (MonoGraph *g);
static void opb (List *wlist, List *opbflips);
static void opc (ChromonList *clist, List *wlist);

__inline__ static void stabilise_neighbours (Node *n, List *fliplist);
__inline__ static Node *opb_check (Node *n);
__inline__ static void process_fliplist (List *fliplist, List *wlist, List *opbflips);

__inline__ static bool opc_check (ChromonList *clist, Node *n);
__inline__ static void process_opbable (List *aflips, List *blist, List *fliplist);
__inline__ static void process_opcable (ChromonList *c, List *flips, List *wlist);
__inline__ static void process_rediscover (List *tot_list, List *fliplist);


/* Interface */
extern ChromonList
*ef_algorithm (MonoGraph *g)
{
	ChromonList *c;
	
	/* We do an operate (a) on Graph g */
	operationa (g);

	/* Operation (b). We also expect this to
	 * restore everything to the operation (a) condition */
	operationb (g);

	/* Operation (c). Again, we expect this to restore
	 * so that operation(a)  and operation(b) hold true
	 * for all vertices */
	c = operationc (g);

	/* We should have a EF'd out graph! */

	return c;
}

/* Operation (a) */
static void
operationa (MonoGraph *g)
{
	gstabilise (g);

	return;
}

/* Operation (b) */
static void 
operationb (MonoGraph *g)
{
	unsigned int	i;
	List		*wlist;

	wlist = create_list ();
	
	/* Ok, add all adjacent, 4-balanced vertices to the wlist */
	for ( i = 0; i < g->vertices; i++ )
		if ( opb_check (g->vlist [i] ) )
				wlist->method->append (wlist, g->vlist [i]);

	opb (wlist, NULL);

	delete_list (wlist);

	/* Postcondition: A stable colouring of g, hopefully more
	 * optimised */

	return;
}



static void
opb (List *wlist, List *opbflips)
{
	Node		*n;
	Node		*m;
	List		*fliplist;

	fliplist = create_list ();

	while ( wlist->length ) {
		/* We check for the op(b) conditions here again, because
		 * previous local operations could have destroyed this */

		
		n = wlist->method->head (wlist);
		m = opb_check (n);

		/* Ok, we do the flip */
		if ( m ) {
			n->method->invert (n);
			n->method->invert (m);
			
			/* Our caller might want to know our op(b)
			 * flips */
			if ( opbflips ) {
				opbflips->method->append (opbflips, n);
				opbflips->method->append (opbflips, m);
			}


			/* When we do in stabilise, we could potentially
			 * create an opportunity for op(b). So we run
			 * stabilise on each co-chromatic neighbour */
			stabilise_neighbours (n, fliplist);
			stabilise_neighbours (m, fliplist);
	
			/* We now process our list for possible candidates
			 * for op (b) from our fliplist */
			process_fliplist (fliplist, wlist, opbflips);
		}
	}

	delete_list (fliplist);
	
	return;
}

/* Return true if the node is a candidate for an operation b */
__inline__ static Node
*opb_check (Node *n)
{
	unsigned int	i;

	/* Not 4-balanced */
	if ( n->neighbours != 4 ) 
		return NULL;
	
	for ( i = 0; i < n->neighbours; i++ ) 
		/* We only bother looking at anti-chromatic neighbours */
		if ( n->neighbour [i]->colour != n->colour )
			/* If it's balanced, then we have a true condition */
			if ( balanced (n->neighbour [i]) ) 
				return n->neighbour [i];

	return NULL;
}

__inline__ static void
stabilise_neighbours (Node *n, List *fliplist)
{
	unsigned int	i;
	
	for ( i = 0; i < n->neighbours; i++ )
		if ( n->colour == n->neighbour [i]->colour ) 
			stabilise (n->neighbour [i], fliplist);
}

__inline__ static void
process_fliplist (List *fliplist, List *wlist, List *opbflips)
{
	Node		*n;
	Node		*m;
	unsigned int 	i;

	
	while ( fliplist->length ) { 
		n = fliplist->method->head (fliplist);
		
		/* Is this current node op(b)'able ? */
		if ( opb_check (n) )
			wlist->method->append (wlist, n);

		/* We check whether any of our co-chromatic
		 * neighbours are op(b)'able. */
		for ( i = 0; i < n->neighbours; i++ )
			if ( n->neighbour [i]->colour == n->colour )
				if ( (m = opb_check (n)) )
					wlist->method->append (wlist, n);

		/* Our caller might want us to return all the items
		 * in fliplist back as well - ie, for maintaining chromonlist */
		if ( opbflips )
			opbflips->method->append (opbflips, n);
	}

	return;
}

/* Operation (c) */
static ChromonList 
*operationc (MonoGraph *g)
{
	ChromonList	*clist;
	List		*wlist;
	unsigned int	i;

	clist = create_chromonlist ();

	/* Precondition: g is stable */
	clist->method->discover (clist, g);

#ifdef __DEBUG__
	printf ("\nPre-Operation (c)\n");
	print_chromonlist (clist);
#endif 

	/* We have to go off and work on this list now */

	wlist = create_list ();

	/* This is the list of 4-balanced vertices in
	 * the graph, in which op(c) is possible */
	for ( i = 0; i < clist->graph->vertices; i++ )
		if ( clist->graph->vlist [i]->neighbours == 4 )
			if ( opc_check (clist, clist->graph->vlist [i]) )
				wlist->method->append (wlist, clist->graph->vlist [i]);

	/* We call op(c) on this initial list */
	opc (clist, wlist);

	delete_list (wlist);

#ifdef __DEBUG__
	printf ("\nPost-Operation (c)\n");
	print_chromonlist (clist);
#endif

	return clist;
}

static void
opc (ChromonList *c, List *wlist)
{
	List		*blist;
	List		*fliplist;
	List		*flips;
	Node		*n;
	
	flips = create_list ();
	blist = create_list ();
	fliplist = create_list ();

	while ( wlist->length ) {
		n = wlist->method->head (wlist);

		if ( opc_check (c, n) ) {
			
			n->method->invert (n);

			/* We need to rebuild chromons. So, we need to maintain
			 * this list */
			fliplist->method->append (fliplist, n);

			/* We could never make the node n unstable by our flip. 
			 * Why? Because it is balanced. Flipping a balanced vertex
			 * makes it remain balanced. But we could potentially make 
			 * our new cochromatic neighbours unbalanced. This
			 * is equivilant to doing a op(a) on the whole graph. */
			stabilise_neighbours (n, flips);

			/* We now expand each node in aflips into
			 * blist to add each op(b)'able neighbours. 
			 * This also adds all of fliplist into aflips. */
			process_opbable (flips, blist, fliplist);
			
			/* Now, we go off and see if we can do op(b). */
			opb (blist, fliplist);

			
			/* We need to copy all our fliplist elements
			 * into flips, with all its neighbours included.
			 * We've upset the local viscinity */
			process_rediscover (flips, fliplist); 
			
			/* Now, our graph at this point is stable. So
			 * we need to rediscover our chromons for the vertices
			 * we've disrupted. rediscover () is non destructive,
			 * so we still have all our flipped elements!!! */
			c->method->rediscover (c, flips);

			/* Now, we need to process our flips list, to see whether we've created
			 * more vertices which are op(c)'able. We also need to maintain
			 * our fliplist, ie copy our flips into fliplist.
			 * This function is destructive, so we lose our fliplist */
			process_opcable (c, fliplist, wlist);
		}
	}

	delete_list (flips);
	delete_list (blist);
	delete_list (fliplist);
}

__inline__ static bool
opc_check (ChromonList *clist, Node *n)
{
	Node		*new [2] = {NULL, NULL};
	unsigned int	i;
	
	if ( !balanced (n) )
		return false;
		
	for ( i = 0; i < 4; i++ )
		if ( n->neighbour [i]->colour != n->colour ) {
			if ( new [0] )
				new [1] = n->neighbour [i];
			else
				new [0] = n->neighbour [i];
		}

	/* If new [0] and new [1] are path connected, ie,
	 * they are in the same chromon, then we only look
	 * at the size of our chromon as 
	 * clist->vertice [new [0]->id].list->length + 1. */
	if ( clist->vertice [new [0]->id].list ==
			clist->vertice [new [1]->id].list )
		return (clist->vertice [ new [0]->id ].list->length + 1) 
			< clist->vertice [n->id].list->length;

	/* THey don't form a circuit. So, what happens? */
	return (clist->vertice [ new [0]->id ].list->length +
			clist->vertice [ new [1]->id ].list->length + 1)
		< clist->vertice [n->id].list->length;
}

__inline__ static void
process_opbable (List *aflips, List *blist, List *fliplist)
{
	Node		*n;
	unsigned int	i;

	while ( aflips->length ) {
		n = aflips->method->head (aflips);

		/* We always add our node to this, because stabilise
		 * would have flipped this */ 
		fliplist->method->append (fliplist, n);

		/* We add this condition in for safety */
		if ( opb_check (n) )
			blist->method->append (blist, n);
		
		for ( i = 0; i < n->neighbours; i++ )
			/* Only cochromatic neighbours pose any risk */
			if ( n->neighbour [i]->colour == n->colour )
				if ( opb_check (n->neighbour [i]) )
					blist->method->append (blist, n->neighbour [i]);
	}

	/* aflips is now empty. blist contains all the possible nodes that
	 * may have become processable by operation (b), from locally stabilising.
	 * fliplist contains all the elements which were flipped. */

	return;
}

__inline__ static void
process_opcable (ChromonList *c, List *flips, List *wlist)
{
	Node		*n;
	unsigned int	i;
	
	while ( flips->length ) {
		n = flips->method->head (flips);

		if ( n->neighbours == 4 )
			if ( opc_check (c, n) )
				wlist->method->append (wlist, n);

		for ( i = 0; i < n->neighbours; i++ )
			/* Surely, we couldn't have only annoyed our
			 * cochromatic neighbours? */
			if (n->neighbour [i]->colour == n->colour) 
				if ( opc_check (c, n->neighbour [i]) )
					wlist->method->append (wlist, n->neighbour [i]);
	}

	return;
}

__inline__ static void
process_rediscover (List *tot_list, List *fliplist)
{
	unsigned int	i;
	Node		*n;
	
	DEBUG_ASSERT (tot_list->length == 0);

	while ( fliplist->length ) {
		n = fliplist->method->head (fliplist);

		for ( i = 0; i < n->neighbours; i++ )
			tot_list->method->append (tot_list, n->neighbour [i]);

		tot_list->method->append (tot_list, n);
	}

	/* Now tot_list contains all the chromons we need to
	 * look at */
	return;
}

/* We call this version "Enhanced EF, or EEF algorithm */
extern ChromonList
*eef_algorithm (MonoGraph *g)
{
	ChromonList *c;
	
	/* We do an operate (a) on Graph g */
	operationa (g);

	/* Operation (b). We also expect this to
	 * restore everything to the operation (a) condition */
	operationb (g);

	/* Operation (c). Again, we expect this to restore
	 * so that operation(a)  and operation(b) hold true
	 * for all vertices */
	c = operationc (g);

	/* We should have a EF'd out graph! */

	/* This is a little inefficient, but we'll do 
	 * this for the time being */
	delete_chromonlist (c);

	/* We try to minimise the chromonlist like in 
	 * the EHST algorithm */
	c = minimise_chromons (g);

	return c;
}

extern ChromonList
*e2ef_algorithm (MonoGraph *g)
{
	ChromonList *c;
	
	/* We do an operate (a) on Graph g */
	operationa (g);

	/* Operation (b). We also expect this to
	 * restore everything to the operation (a) condition */
	operationb (g);

	/* Operation (c). Again, we expect this to restore
	 * so that operation(a)  and operation(b) hold true
	 * for all vertices */
	c = operationc (g);

	/* We should have a EF'd out graph! */

	/* This is a little inefficient, but we'll do 
	 * this for the time being */
	delete_chromonlist (c);

	/* We try to minimise the chromonlist like in 
	 * the EHST algorithm */
	c = minimise_chromons (g);

	delete_chromonlist (c);

	c = ef_algorithm (g); 

	return c;
}
