/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    generic.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <algorithm/algorithm.h>

__inline__ static bool flipcheck (ChromonList *clist, Node *n);

/* Returns true if stable */
extern bool
stable (Node *n)
{
	unsigned int	i;
	unsigned int 	cochromatic = 0;	/* Number of cochromatic neighbours */
	unsigned int 	antichromatic = 0;	/* Number of antichromatic neighbours */

	for ( i = 0; i < n->neighbours; i++ ) {
		if ( n->neighbour [i]->colour == n->colour )
			cochromatic++;
		else 
			antichromatic++;
	}

	/* True iff the neighbours arn't of the same colour */
	return  antichromatic >= cochromatic;
}

/* Returns true if 4-balanced */
extern bool
balanced (Node *n)
{
	unsigned int 	i;
	unsigned int 	colour_count;

	/* We can't ever have a node that is 4-balanced without
	 * 4 neighbours!!! */
	if ( n->neighbours != 4 ) return false;
	
	for ( i = 0, colour_count = 0; i < 4; i++ )
		/* Choice of colour is arbitary. We're looking for balance! */
		if ( n->neighbour [i]->colour == BLACK )
			colour_count++;

	return colour_count == 2;
}

extern bool
balanced2 (Node *n)
{
	/* We only care about nodes with 2 neighbours */
	if ( n->neighbours != 2 ) return false;

	/* If the two colours are not equal, then we must
	 * have a 2-balanced node */
	return n->neighbour [0] != n->neighbour [1];
}

/* Precondition: A graph with an arbitary assignment.
 * Postcondition: A stable assignment for the graph. */
extern void
gstabilise (MonoGraph *g)
{
	unsigned int	i;	
	bool 		touched = false;

	
	/* Precondition: We expect a graph g with degree <= 4 */
	do {
		for ( i = 0, touched = false; i < g->vertices; i++ )
			if ( !stable ( g->vlist [i] ) ) {
				/* Mark graph touched */
				touched = true;
				g->vlist [i]->method->invert ( g->vlist [i] );
			}
	} while ( touched );

	/* Postcondition: A stable colouring for g */

	return;
}

/* We provide this for debugging */
extern void
print_chromonlist (ChromonList *clist)
{				
		List	*l;		
		List	*w;		
		unsigned int	i;	
		unsigned int 	j;	
					
					
		printf ("Black:\n");	
		for ( w = clist->black; ;					
				printf ("White:\n"), w = clist->white ) {	
			w->method->reset (w);
			for ( i = 0; i < w->length; i++ ) {			
				l = w->method->next (w);			
				l->method->reset (l);
										
				printf ("[ ");					
				for ( j = 0; j < l->length; j++) 		
					printf ("%d ", ((Node *) l->method->next (l))->id );	
				printf ("]\n");					
			}							
			if ( w == clist->white )				
				break;						
		}								
}

extern unsigned int
cochromatic (Node *n)
{
	unsigned int 	i;
	unsigned int	count;

	for ( i = 0, count = 0; i < n->neighbours; i++ )
		if ( n->neighbour [i]->colour == n->colour )
			count++;

	return count; 
		
}

/* Flip & Stabilise */

extern void
flip_and_stabilise (Node *n, List *fliplist)
{
	unsigned int	i;

	DEBUG_ASSERT ( n != NULL );

	/* We flip */
	n->method->invert (n);

	if ( fliplist )
		fliplist->method->append (fliplist, n);

	/* Then we stabilise all cochromatic neighbours */
	for ( i = 0; i < n->neighbours; i++ )
		if ( n->colour == n->neighbour [i]->colour )
			stabilise (n->neighbour [i], fliplist);

	return;
}

extern void
stabilise (Node *n, List *fliplist)
{
	List 		*l;
	unsigned int 	i;

	l = create_list ();

	DEBUG_ASSERT ( n != NULL );

	/* This is slightly slower, because we know
	 * we have to flip n. But it really doesn't matter,
	 * and this is much more elegant */
	l->method->append (l, n);

	while ( l->length ) {
		n = l->method->head (l);

		/* Check for stability */
		if ( !stable (n) ) {
		
			n->method->invert (n);

			/* We notify that we've added the flipped
			 * vertice to the fliplist */
			if ( fliplist )	
				fliplist->method->append (fliplist, n);
			

			/* Add cochromatic neighbours to the list */ 
			for ( i = 0; i < n->neighbours; i++ )
				if ( n->colour == n->neighbour [i]->colour )
					l->method->append (l, n->neighbour [i]);
		}
	}
	
	/* We're through using the list */
	delete_list (l);

	return;
}

extern ChromonList
*minimise_chromons (MonoGraph *g)
{
	ChromonList 	*c;
	unsigned int	i, j;
	List		*fliplist;

	c = create_chromonlist ();
	c->method->discover (c, g);

	fliplist = create_list ();

	/* Now that we have a chromonlist, we go off
	 * and check all vertices */

	for ( i = 0; i < g->vertices; i++ )
		if ( flipcheck (c, g->vlist [i]) ) {
			g->vlist [i]->method->invert (g->vlist [i]);

			fliplist->method->append (fliplist, g->vlist [i]);
			
			/* Add neighbours */
			for ( j = 0; j < g->vlist [i]->neighbours; j++ )
				fliplist->method->append (fliplist, g->vlist [i]->neighbour [j]);
			
			c->method->rediscover (c, fliplist);
		}

	return c;
}
__inline__ static bool
flipcheck (ChromonList *clist, Node *n)
{
	Node		*antichromatic [4] = {NULL, NULL, NULL, NULL};
	unsigned int	i, j, k;

	for ( i = 0; i < n->neighbours; i++ )
		if ( n->neighbour [i]->colour != n->colour ) { 
			for ( j = 0; antichromatic [j] && j < 4; j++ ) 
				;

			antichromatic [j] = n->neighbour [i];
		}

	/* Set j */
	for ( j = 0; antichromatic [j] && j < 4; j++ )
		;
	
	
	/* Check for any vertices which are in the same chromon
	 * neighbouring n */
	for ( i = 0; i < j; i++ )
		for ( k = i + 1; k < j; k++ ) 
			/* Circular chromon */
			if ( (clist->vertice [antichromatic [i]->id].list 
						== clist->vertice [antichromatic [k]->id].list)) { 
				antichromatic [k] = NULL; 
				antichromatic [k] = antichromatic [--j];
				antichromatic [j] = NULL;
			}
	/* Look at our new chromon if we flip */
	for ( k = 0, i = 0; i < j; i++ )
		k += clist->vertice [antichromatic [i]->id].list->length;

	/* True if our new chromon is smaller! */
	return k < clist->vertice [n->id].list->length;
}
