/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    hst.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <algorithm/hst.h>

/* Internal functions */
static void whitify (MonoGraph *g);
static List *get_blacks (MonoGraph *g);
static void deliberate_assignment (MonoGraph *g, List *blacks);

/* Step (4) Prototypes */
__inline__ static bool unripen (Node *n, List *flist);
__inline__ static Node *ripe (Node *n);
__inline__ static int count_neighbours (Node *bal [4],
		unsigned int ncount [4], unsigned int k1);
__inline__ static void unbalancer (MonoGraph *g, List *flist);
__inline__ static bool wbalanced (Node *n);
__inline__ static void eval_list (List *wlist, List *flist);


/* Step (5) Prototypes */
__inline__ static void work_on_black (MonoGraph *g, 
		bool *dvlist, List *list, unsigned int v);
__inline__ static void pick_right_blacks (List *blacks, List *chromon);
__inline__ static bool circuit (List *chromon);
__inline__ static void work_on_chromon (List *blacks, List *chromon);
__inline__ static Node *path_rewind (Node *cnode);


/* Step (6) Prototypes */
__inline__ static void work_on_graph (MonoGraph *g, MonoGraph *h1, 
		MonoGraph *h1p, unsigned int *blacklist, 
		unsigned int blacklength, Colour c);
__inline__ static bool distance (Node *n, Node *m, Colour c);
__inline__ static void make_directed_graph (MonoGraph *g);
__inline__ static void give_direction (Node *start, bool *done);
__inline__ static void do_assignments (MonoGraph *g, MonoGraph *h1, 
		MonoGraph *h1p, MonoGraph *h2, MonoGraph *h2p, 
		unsigned int *blacklist, unsigned int blacklength);
__inline__ static unsigned int unplaced_successor (MonoGraph *h, 
		unsigned int u, bool *done);
__inline__ static unsigned int get_undone_vertex (bool *done, unsigned int l);
__inline__ static void colour_white (List *l);

/* Enhanced HST code */ 
static void ewhitify (MonoGraph *g);
static void edeliberate_assignment (MonoGraph *g, List *blacks);
__inline__ static void edo_assignments (MonoGraph *g, MonoGraph *h1, 
		MonoGraph *h1p, MonoGraph *h2, MonoGraph *h2p, 
		unsigned int *blacklist, unsigned int blacklength);

#ifdef __DEBUG__

/* Debugger routines */
static void print_list (List *l);
static void print_neighbours (Node *n);
#endif /* __DEBUG__ */



/* The algorithm caller */
extern ChromonList 
*hst_algorithm (MonoGraph *g)
{
	ChromonList	*c;
	List		*blacks;
	
	/* We firstly create a stable colouring */
	gstabilise (g);

	/* Now we have to do step (4) - ie make all white
	 * chromons with lengths <= 4 */
	whitify (g);

#ifdef __DEBUG__
	c = create_chromonlist ();
	c->method->discover (c, g);

	print_chromonlist (c);

	delete_chromonlist (c);
	
	/* We check for any unbalanced vertices */
	{
		unsigned int	i;

		for ( i = 0; i < g->vertices; i++ ) {
			if ( cochromatic (g->vlist [i]) > 2 )
				printf ("unbalanced vertice %d\n", i);
		}

		/* We test for any ripe/white balanced vertices */
		for ( i = 0; i < g->vertices; i++ ) {
			if ( ripe (g->vlist [i]) )
				printf ("ripe vertex: %d", i);
			else if ( (g->vlist [i]->colour == WHITE)  
					&& balanced (g->vlist [i]) )
				printf ("white balanced vertex: %d", i);
		}
	}
#endif
	/* This is step (5) of the algorithm.
	 * The list blacks contains the maximum independant
	 * set of balanced black vertices such that no two are
	 * ajacent */
	blacks = get_blacks (g);

#ifdef __DEBUG__
	print_list (blacks);
#endif

	/* Step (6) */
	deliberate_assignment (g, blacks);

	delete_list (blacks);

	/* Now we build our chromonlist */
	c = create_chromonlist ();
	c->method->discover (c, g);

#ifdef __DEBUG__ 
	printf ("Post Step (6):\n");
	print_chromonlist (c);
#endif 

	return c;
}


/* Implementation of step (4) of the algorithm.
 * This expects a stable colouring. */
static void
whitify (MonoGraph *g)
{
	unsigned int	i;
	List		*wlist;
	List		*flist;
	Node		*n;

	/* Keep track of potential vertices on */
	wlist = create_list ();
	/* Keep track of flipped vertices */
	flist = create_list ();
	
	/* Initially, we add all the vertices into wlist */
	for ( i = 0; i < g->vertices ; i++ ) 
		wlist->method->append (wlist, g->vlist [i]);

	/* Alternatively: */
	
#if 0
	for ( i = 0; i < g->vertices ; i++ )
		unripen (g->vlist [i], flist);

	eval_list (wlist, flist);
#endif 
	
	/* We keep doing this until our working list is
	 * nonempty */
	while ( wlist->length ) {

		/* Easy! */
		while ( wlist->length ) {
			n = wlist->method->head (wlist);

			if ( unripen (n, flist) )
				;

			eval_list (wlist, flist);
		}

		/* By this stage, we have a graph with 
		 * no ripe vertices. We go off and get rid
		 * of the white balanced vertices */
		unbalancer (g, flist);

		eval_list (wlist, flist);
	}

	delete_list (wlist);
	delete_list (flist);

	return;
}



#if 0

/* This is the brute force way of doing things! */
static void
whitify (MonoGraph *g)
{
	unsigned int	i;
	bool		ripe;

	do {
		do {
			ripe = false;
			
			for ( i = 0; i < g->vertices; i++ )
				if ( unripen (g->vlist [i], NULL) )
					ripe = true;
			
		} while ( ripe );

		/* ripe = false */
		
		/* White and balanced */
		for ( i = 0; i < g->vertices; i++ )
			if ( wbalanced (g->vlist [i]) ) {
				flip_and_stabilise (g->vlist [i], NULL);
				ripe = true;
				break;
			}
		
		/* If we have no ripe & white balanced vertices,
		 * we come here with ripe = false. */
	} while ( ripe );

	return;
	
}
#endif 

__inline__ static bool
unripen (Node *n, List *flist)
{
	Node	*r;

	r = ripe (n);

	if ( !r )
		return false;

	/* We keep doing this until the vertice becomes 
	 * unripe */
	do {
		flip_and_stabilise (r, flist);
		
	} while ( (r = ripe (n)) );

	return true;
}

__inline__ static Node
*ripe (Node *n)
{
	unsigned int	k1, k3;
	unsigned int	i;
	
	/* Maintaining balanced vertices */
	Node		*bal [4];
	unsigned int	ncount [4] = {0, 0, 0, 0};

	/* We possibly cannot have a ripe vertex with 2 neighbours
	 * or less. */
	if ( n->neighbours <= 2 )
		return NULL;
	
	for ( i = 0, k1 = 0, k3 = 0; i < n->neighbours; i++ ) {
		/* Anti-chromatic */
		if ( n->neighbour [i]->colour != n->colour ) {
			/* 4 Balanced */
			if ( balanced (n->neighbour [i]) )
				bal [k1++] = n->neighbour [i];
			/* 2 balanced */
			else if ( balanced2 (n->neighbour [i]) )
				bal [k1++] = n->neighbour [i]; 
		}
		/* Co-chromatic */
		else 
			k3++;
	}

	/* If this is the case, then we have k2 > 2. So, our
	 * inequality cannot hold */
	if ( k1 + k3 <= 2 )
		return NULL;

	/* Now k1 is the number of balanced vertices, with bal 
	 * an array pointing to these. ncount [i] is the number
	 * of ajacent neighbours bal [i] has with some other node
	 * in bal [i]. */

	i = count_neighbours (bal, ncount, k1);

	while ( ncount [i] ) {
		/* Now, if we got here, i should contain
		 * the index of the largest number of neighbours
		 * that a balanced vertex has. We remove this vertex
		 * from our bal. */

		/* So, here, we overwrite our ith element with our k1
		 * element. */
		bal [i] = bal [--k1];

		/* This may make things slightly efficient */
		if (k1 + k3 <= 2)
			return NULL;

		i = count_neighbours (bal, ncount, k1);
	}


	/* If this equality is true, then we must have at least
	 * one element bal. So, we simply return the top one, because
	 * it's arbitary. */
	if ( (k1 + k3) > 2 )
		return bal [0];
	
	return NULL;
}


__inline__ static int
count_neighbours (Node *bal [4], unsigned int ncount [4], unsigned int k1)
{
	unsigned int	i, j, k;
	unsigned int	highest = 0;

	for ( i = 0; i < k1; i++ )
		ncount [i] = 0;
	
	for ( i = 0; i < k1; i++ ) {
		for ( j = 0; j < k1; j++ )
			for ( k = 0; k < bal [i]->neighbours; k++ )
				/* Count the number of neighbours */
				if ( bal [i]->neighbour [k] == bal [j] )
					ncount [i]++;

		/* We record the highest seen ncount */
		if ( ncount [i] > ncount [highest] )
			highest = i;
	}

	/* Now ncount [i] contains the number of ajacent neighbouring
	 * nodes with bal [i] */
	return highest;
}
__inline__ static void
unbalancer (MonoGraph *g, List *flist)
{
	unsigned int		i;
	
	for ( i = 0; i < g->vertices; i++ )
		/* Each time we  do this, we go back to the
		 * This is a bit slow, we need to optimise at some
		 * point */
		if ( wbalanced (g->vlist [i]) ) {
			flip_and_stabilise (g->vlist [i], flist);
			return;
		}

	return;
}

__inline__ static bool
wbalanced (Node *n)
{
	/* We must have colour WHITE and degree = 4 */
	if ( n->neighbours < 4 || n->colour == BLACK )
		return false;

	/* We have a hit */
	if ( balanced (n) )
		return true;

	return false;
}

__inline__ static void
eval_list (List *wlist, List *flist)
{
	unsigned int	i, j;
	Node		*n;

	while ( flist->length )  {
		n = flist->method->head (flist);

		/* Add this vertice */
		wlist->method->append (wlist, n);
		/* Now add the neighbours */
		for ( i = 0; i < n->neighbours; i++) {
			wlist->method->append (wlist, n->neighbour [i]);

			/* We also need to add their neighbours, by the
			 * nature of this problem. Ie flipping a vertex
			 * could make a vertex that is of distance 2
			 * ripe. */
			for ( j = 0; j < n->neighbour [i]->neighbours; j++ )
				wlist->method->append (wlist, 
						n->neighbour [i]->neighbour [j]);
		}
	}

	return;
}


/* Step (5) of the algorithm. */
static List
*get_blacks (MonoGraph *g)
{
	List		*blacks;
	List		*chromon;
	bool		*dvlist;
	unsigned int	i;

	blacks = create_list ();
	chromon = create_list ();

	/* This is to keep track of whether we've added a vertice in. */
	dvlist = (bool *) malloc (sizeof (bool) * g->vertices);

	/* Set it to false initially */
	memset (dvlist, (int) false, sizeof (bool) * g->vertices);

	for ( i = 0; i < g->vertices; i++ )
		if ( g->vlist [i]->colour == BLACK )
			/* We havn't already worked on the
			 * chromon this belongs to. */
			if ( !dvlist [i] ) {
				work_on_black (g, dvlist, chromon, i);
#ifdef __DEBUG__ 
				printf ("Chromon: ");
				print_list (chromon);
#endif /* __DEBUG__ */ 
				pick_right_blacks (blacks, chromon);
			}


	return blacks;
}



/* This expects chromon to be an active object */
__inline__ static void
work_on_black (MonoGraph *g, bool *dvlist, List *chromon, unsigned int v)
{
	Node		*fnode;
	Node		*n;
	List		*process_list; /* List to know which to process */
	unsigned int	i;

	/* First node of the chromon */
	fnode = path_rewind (g->vlist [v]);

	process_list = create_list ();

	/* We handle the base case separately. This way, we can build circuits
	 * in order too!!! */
	chromon->method->append (chromon, fnode);
	dvlist [fnode->id] = true;
	
	for ( i = 0; i < fnode->neighbours; i++ )
		/* We only add one. We can guarentee that this will pick
		 * build the chromon correctly because fnode is the start
		 * node for a path. If it is a circuit, then it'll come back
		 * to this. */
		if ( fnode->neighbour [i]->colour == fnode->colour ) { 
			process_list->method->append (process_list, fnode->neighbour [i]);
			break;
		}
	

	/* We go off and build our chromon */
	while ( process_list->length ) {
		n = process_list->method->head (process_list);

		/* We haven't already allocated this vertice */
		if ( !dvlist [n->id] ) { 
			for ( i = 0; i < n->neighbours; i++ )
				/* We need to go off and process our co-chromatic
				 * neighbours */
				if ( n->neighbour [i]->colour == n->colour )
					process_list->method->append (process_list, n->neighbour [i]);

			chromon->method->append (chromon, n);

			dvlist [n->id] = true;
		}
	}

	/* We have no use for this */
	delete_list (process_list);

	return;
}

__inline__ static Node
*path_rewind (Node *cnode)
{
	Node		*left;
	Node		*cur;
	unsigned int	i;
	bool 		term;


	/* Determine "left" and "right" vertices.
	 * The direction is chosen arbitarily. (It infact doesn't
	 * matter as long as we're consistent. */
	for ( i = 0, left = NULL;
			i < cnode->neighbours; i++ ) 
		if ( cnode->neighbour [i]->colour == cnode->colour ) {
			left = cnode->neighbour [i];
			break;
		}

	/* For consistency for below */
	cur = NULL;

	/* Base case for going left. One of the vertices will undoubtedly
	 * be cnode. So we ignore cnode. */
	if ( left ) {
		for ( i = 0, cur = left; i < cur->neighbours; i++ )
			/* We want our "left" node to be different from
			 * cnode, which we know is the neighbour of cur */
			if ( (cur->neighbour [i] != cnode) && 
					(cur->colour == cur->neighbour [i]->colour) )  {
				
				/* These are the initial values for the
				 * general case */
				left = cur;
				cur = cur->neighbour [i];

				/* Only a path or circuit */
				break;
			}

		/* Either we will have left = NULL, or we have
		 * a true left which is different from cnode */
	}
	
	/* End of the road? */
	if ( cur == left ) {
		if ( !cur ) 
			return cnode;
		else 
			return cur;
	}

	/* We add the vertices to the "left" */
	for ( term = false; !term && cur; ) {
		
		for ( i = 0, term = true; i < cur->neighbours; i++ ) {
			if ( cur->neighbour [i]->colour == cur->colour ) {
				/* Terminate, we have a circular subgraph. We 
			 	* terminate, but also prevent travasing the "right". */
				
				if ( cur->neighbour [i] == cnode ) 
					return cnode;

				/* Bingo! We have the only other neighbour. We
				 * only terminate when term is not changed, ie 
				 * we've exhausted our possibilities */
				else if ( cur->neighbour [i] != left ) {
					left = cur;
					cur = cur->neighbour [i];
	
					term = false;
	
					break;
				}
			}
		}
	}

	DEBUG_ASSERT ( cur != NULL );
	return cur;
}

__inline__ static void
pick_right_blacks (List *blacks, List *chromon)
{
	/* We have only one or two elements in chromon - so we
	 * don't add, because our blacks list can't contain
	 * start or end of the path */
	if ( chromon->length == 1 || chromon->length == 2) {
		while ( chromon->length )
			chromon->method->head (chromon);

		return;
	}

	/* Now, it becomes interesting. We determine whether
	 * the chromon we have is a path or circuit */
	if ( circuit (chromon) )
		work_on_chromon (blacks, chromon);
	else { 
		chromon->method->head (chromon);
		work_on_chromon (blacks, chromon);
	}

	/* Chromon should be empty now */
	DEBUG_ASSERT (chromon->length == 0); 

	return;
}

__inline__ static bool
circuit (List *chromon)
{
	Node		*n;
	unsigned int	i;

	n = chromon->last->data;

	for ( i = 0; i < n->neighbours; i++ ) {
		/* Path condition */
		if ( n->neighbour [i] == chromon->root->data )
			return true;
	}

	return false;
}

__inline__ static void
work_on_chromon (List *blacks, List *chromon)
{
	Node		*n;

	/* We have two conditions for path lengths even and odd */
	while ( chromon->length > 1 ) {
		n = chromon->method->head (chromon);
		chromon->method->head (chromon);

		blacks->method->append (blacks, n);
	}

	if ( chromon->length )
		chromon->method->head (chromon);

	return;
}


/* Step (6) */

static void
deliberate_assignment (MonoGraph *g, List *blacks)
{
	/* We create 4 graphs, each with blacks->length
	 * number of nodes */
	MonoGraph	*h1, *h2;
	MonoGraph	*h1p, *h2p;
	unsigned int	*blacklist;
	unsigned int	blacklength;

	/* In this case, we have an optimal
	 * colouring by HST standards. */
	if ( !blacks->length )
		return; 
	
	/* Sucessors */ 
	h1 = create_graph (blacks->length);
	h2 = create_graph (blacks->length);
	/* Predecessors */
	h1p = create_graph (blacks->length);
	h2p = create_graph (blacks->length);

	/* We keep a mapping table between the vertices
	 * in h1, h2, h1p and h2p, and the vertice numbers of
	 * the elements in blacks. */
	blacklist = (unsigned int *) malloc (sizeof (unsigned int *) * blacks->length);

	for ( blacklength = 0; blacks->length; blacklength++ )
		blacklist [blacklength] = ((Node *) (blacks->method->head (blacks)))->id;
	
	/* Now we create the gray and white graph */
	work_on_graph (g, h1, h1p, blacklist, blacklength, WHITE);
	
	/* Now we create the gray and white graph */
	work_on_graph (g, h2, h2p, blacklist, blacklength, BLACK);


	/* We have h1, h2: a di-graph, with h1p, h2p and
	 * oppositely directioned graph. So, we decide
	 * on the colouring */
	
	do_assignments (g, h1, h1p, h2, h2p, blacklist, blacklength);

	/* Remeber to deallocate */
	delete_graph (h1);
	delete_graph (h2);
	delete_graph (h1p);
	delete_graph (h2p);

	free (blacklist);
	
	return;
}

__inline__ static void
work_on_graph (MonoGraph *g, MonoGraph *h, MonoGraph *hp, 
		unsigned int *blacklist, unsigned int blacklength, Colour c)
{
	unsigned int	i, j;
	
	/* We put an edge between two vertices in h, iff
	 * they are distance 2 or 3 across white vertices in g.
	 * We choose directions later. */
	for ( i = 0; i < blacklength; i++ )
		for ( j = i + 1; j < blacklength; j++ )
			if ( (i != j) &&  distance (g->vlist [blacklist [i]],
						g->vlist [blacklist [j]], c) )
				h->method->uconnect (h, i, j);

	make_directed_graph (h);

	/* We create hp, which is a graph of predecessors of h */
	for (i = 0; i < h->vertices; i++ )
		for ( j = 0; j < h->vlist [i]->neighbours; j++ )
			/* We would otherwise have a vertice going from
			 * i to h->vlist [i]->neighbour [j]->id. We
			 * switch this around for hp */
			hp->method->connect (hp, 
					h->vlist [i]->neighbour [j]->id, i);

	/* Now h contains a directed graph consisting of either
	 * disjoint paths or cycles. hp contains the same, except that
	 * each path or cycle is going in the opposite direction to
	 * that of h. From the symmetry, we could obviously choose
	 * hp as h. */

	return;
}

__inline__ static bool
distance (Node *n, Node *m, Colour c)
{
	Node		*n_neighbour [4];
	Node		*m_neighbour [4];
	unsigned int	ncount, mcount;
	unsigned int	i, j, k;

	/* Grab neighbours of colour c from n */ 
	for ( i = 0, ncount = 0; i < n->neighbours; i++ ) 
		if ( n->neighbour [i]->colour == c )
			n_neighbour [ncount++] = n->neighbour [i];

	/* Grab neighbours of colour c from m */
	for ( i = 0, mcount = 0; i < m->neighbours; i++ )
		if ( m->neighbour [i]->colour == c ) 
			m_neighbour [mcount++] = m->neighbour [i];

	/* If they have a distance of 2, then they _must_ have a
	 * common neighbour. Our search is bounded by O (1) */
	for ( i = 0; i < ncount; i++ )
		for ( j = 0; j < mcount; j++ )
			if ( n_neighbour [i] == m_neighbour [j] )
				return true;

	/* If they have a distance of 3, then one of their neighbours'
	 * must point to the neighbour of the other vertice. */
	for ( i = 0; i < ncount; i++ )
		for ( k = 0; k < n_neighbour [i]->neighbours; k++ )
			for ( j = 0; j < mcount; j++ )
				if ( n_neighbour [i]->neighbour [k] == m_neighbour [j] )
					return true;

	return false;
}


__inline__ static void
make_directed_graph (MonoGraph *g)
{
	bool		*done;
	Node		*fnode;
	unsigned int	i;
	
	/* We keep track of stuff we've already done!  */
	done = (bool *) malloc (sizeof (bool) * g->vertices );
	memset (done, (int) false, sizeof (bool) * g->vertices );

	for ( i = 0; i < g->vertices; i++ ) 
		/* We only work on the vertice, if we havn't 
		 * already worked on it. */
		if ( !done [i] ) {
			fnode = path_rewind (g->vlist [i]);
			give_direction (fnode, done);
		}

	free (done);
	
	return;
	
}

__inline__ static void
give_direction (Node *start, bool *done)
{
	Node		*a;
	Node		*b;

	/* Ok, so we have an edge (a,b) and edge (b,a).
	 * So, what we do is to remove the edge (b,a)
	 * from g */

	if ( done [start->id] )
		return;
	
	if ( !start->neighbours ) {
		done [start->id] = true;
		return;
	}

	/* Now, if start is in a path, then start will
	 * only have one neighbour. If it's in a cycle
	 * then it'll have two. But we're giving directions
	 * arbitarily, so we can simply choose neighbour [0]. */
	a = start;
	b = start->neighbour [0];

	/* The start is automatically done */
	done [start->id] = true;

	while (true) {
		/* This is to deal with cycles */
		if (b == start)
			break;

		/* We remove a from b */
		b->method->del_neighbour (b, a);
		done [b->id] = true;

		/* Terminating condition for a path -
		 * if b is the end vertex of a directed
		 * path, then b will have 0 neighbours */
		if ( !b->neighbours ) {
			done [b->id] = true;
			break;
		}

		/* Now we move to the next one */
		a = b; 
		b = b->neighbour [0];
	}

	return;
}

__inline__ static void
do_assignments (MonoGraph *g, MonoGraph *h1, MonoGraph *h1p,
		MonoGraph *h2, MonoGraph *h2p, unsigned int *blacklist,
		unsigned int blacklength)
{
	unsigned int	count;
	unsigned int	i, u, w;
	bool		*done;
	List		*x [2];
	MonoGraph	*h [2];
	MonoGraph	*hp [2];

	/* We colour  x [0] black, and x [1] white */
	x [0] = create_list ();
	x [1] = create_list ();

	h [0] = h1, h [1] = h2;
	hp [0] = h1p, hp [1] = h2p;

	/* This is to indicate what is done and what is not.
	 * It's much quicker than using a list. */
	done = (bool *) malloc (sizeof (bool) * blacklength);
	memset (done, (int) false, sizeof (bool) * blacklength);

	/* u and w are always going to be the 
	 * vertex numbers of the vertices in h [i] graphs,
	 * and never in g */
	u = get_undone_vertex (done, blacklength);
	x [0]->method->append (x [0], g->vlist [ blacklist [u] ]);
	count = blacklength - 1;
	i = 1;
	
	while ( count ) {
		if ( (w = unplaced_successor (h [i - 1], u, done) ) < blacklength ) { 
			i = 3 - i;
			done [w] = true;
			/* This is the opposite colour now, because we've
			 * done a i = 3 - i */
			x [i - 1]->method->append (x [i - 1], g->vlist [ blacklist [w] ]);
			count--;
			
			u = w;
		}
		else if ( ( w = unplaced_successor ( hp [i - 1], u, done) ) < blacklength ) { 
			i = 3 - i;
			done [w] = true;

			/* Opposite colour, we've already flipped i */
			x [i - 1]->method->append (x [i - 1], g->vlist [ blacklist [w] ]);
			count--;

			u = w;
		}
		else { 
			u = get_undone_vertex (done, blacklength); 
			x [0]->method->append (x [0], g->vlist [ blacklist [u] ]);
			count--;
			
			i = 1;
		}
	}

	/* Now we go and do the colourings. We need not worry
	 * about x [1] because we expect all the nodes to be black anyway. */
	colour_white (x [0]);

	delete_list (x [0]);
	delete_list (x [1]);

	free (done);

	return;
	
}

__inline__ static unsigned int
unplaced_successor (MonoGraph *h, unsigned int u, bool *done)
{
	/* done should be of length h->vertices */
	if ( !h->vlist [u]->neighbours )
		return h->vertices;

	/* This is neccessarily the successor of this vertex.
	 * We check whether we've already added this to some 
	 * other. */
	if ( !done [ h->vlist [u]->neighbour [0]->id]  ) 
		return h->vlist [u]->neighbour [0]->id;

	/* This is if the condition above is false */
	return h->vertices;
}

__inline__ static unsigned int
get_undone_vertex (bool *done, unsigned int l)
{
	unsigned int	i;

	for ( i = 0; i < l; i++ )
		if ( !done [i] )
			break;
	
	DEBUG_ASSERT ( i < l );
	
	/* We will always have a match. Because count != 0
	 * ==> there exists an undone vertex */
	done [i] = true;
	return i;
}

__inline__ static void
colour_white (List *l)
{
	Node		*n;
	
	while ( l->length ) {
		n = l->method->head (l);

		DEBUG_ASSERT ( n->colour == BLACK );

		n->colour = WHITE;
	}

	return;
}


#ifdef __DEBUG__

static void
print_list (List *l)
{
	Node		*n = NULL;

	n = l->method->next (l);

	printf ("Blacklist:\n");
	printf ("[ ");
	while ( n ) {
		printf ("%d ", n->id);

		n = l->method->next (l);
	}
	printf ("]\n");

	return;
}

static void
print_neighbours (Node *n)
{
	unsigned int 	i, j;

	for ( i = 0; i < n->neighbours; i++ ) {
		/* colour of our node */
		printf ("neighbour %d: ", n->neighbour [i]->id );
		if ( n->neighbour [i]->colour == BLACK )
			printf ("B");
		else 
			printf ("W");

		printf ("\n");
		
		for (j = 0; j < n->neighbour [i]->neighbours; j++ ) {
			printf ("\t neighbour %d: ", n->neighbour [i]->neighbour [j]->id );
		
			if ( n->neighbour [i]->neighbour [j]->colour == BLACK )
				printf ("B");
			else 
				printf ("W");

			printf ("\n");
			
		}
	}

	fflush (stdout);
	

	return;
}
#endif /* __DEBUG__ */ 





/* Enhanced HST Code */

extern ChromonList 
*ehst_algorithm (MonoGraph *g)
{
	ChromonList	*c;
	List		*blacks;
	
	/* We firstly create a stable colouring */
	gstabilise (g);

	/* Now we have to do step (4) - ie make all white
	 * chromons with lengths <= 4 */
	ewhitify (g);

#ifdef __DEBUG__
	c = create_chromonlist ();
	c->method->discover (c, g);

	print_chromonlist (c);

	delete_chromonlist (c);
	
	/* We check for any unbalanced vertices */
	{
		unsigned int	i;

		for ( i = 0; i < g->vertices; i++ ) {
			if ( cochromatic (g->vlist [i]) > 2 )
				printf ("unbalanced vertice %d\n", i);
		}

		/* We test for any ripe/white balanced vertices */
		for ( i = 0; i < g->vertices; i++ ) {
			if ( ripe (g->vlist [i]) )
				printf ("ripe vertex: %d", i);
			else if ( (g->vlist [i]->colour == WHITE)  
					&& balanced (g->vlist [i]) )
				printf ("white balanced vertex: %d", i);
		}
	}
#endif

	/* This is step (5) of the algorithm.
	 * The list blacks contains the maximum independant
	 * set of balanced black vertices such that no two are
	 * ajacent */
	blacks = get_blacks (g);

#ifdef __DEBUG__
	print_list (blacks);
#endif

	/* Step (6) */
	edeliberate_assignment (g, blacks);

	delete_list (blacks);

	/* Lashi's Step - We go off and look 
	 * at all the vertices. We flip iff flipping
	 * will reduce the chromon size. But I don't think we 
	 * are now going to be guarenteed that the chromons are
	 * paths or circuits */

#ifdef __DEBUG__ 
	c = create_chromonlist ();
	c->method->discover (c, g); 
	printf ("Post Step (6):\n");
	print_chromonlist (c);

	delete_chromonlist (c);
#endif 

	c = minimise_chromons (g);

#ifdef __DEBUG__ 
	printf ("Post Step (7):\n");
	print_chromonlist (c);
#endif 


	return c;
}

static void
ewhitify (MonoGraph *g)
{
	unsigned int	i;
	List		*wlist;
	List		*flist;
	Node		*n;

	/* Keep track of potential vertices on */
	wlist = create_list ();
	/* Keep track of flipped vertices */
	flist = create_list ();
	
	/* Initially, we add all the vertices into wlist */
	for ( i = 0; i < g->vertices ; i++ ) 
		wlist->method->append (wlist, g->vlist [i]);

	while ( wlist->length ) { 
		n = wlist->method->head (wlist);

		/* See if this is ripe. If so, we unripen
		 * it and return true. */
		if ( unripen (n, flist) )
			;
		/* If the node is white and balanced, then we
		 * flip & stabilise n */
		else if ( wbalanced (n) )
			flip_and_stabilise (n, flist);

		eval_list (wlist, flist);
	}

	delete_list (wlist);
	delete_list (flist);

	return;
}

static void
edeliberate_assignment (MonoGraph *g, List *blacks)
{
	/* We create 4 graphs, each with blacks->length
	 * number of nodes */
	MonoGraph	*h1, *h2;
	MonoGraph	*h1p, *h2p;
	unsigned int	*blacklist;
	unsigned int	blacklength;

	/* In this case, we have an optimal
	 * colouring by HST standards. */
	if ( !blacks->length )
		return; 
	
	/* Sucessors */ 
	h1 = create_graph (blacks->length);
	h2 = create_graph (blacks->length);
	/* Predecessors */
	h1p = create_graph (blacks->length);
	h2p = create_graph (blacks->length);

	/* We keep a mapping table between the vertices
	 * in h1, h2, h1p and h2p, and the vertice numbers of
	 * the elements in blacks. */
	blacklist = (unsigned int *) malloc (sizeof (unsigned int *) * blacks->length);

	for ( blacklength = 0; blacks->length; blacklength++ )
		blacklist [blacklength] = ((Node *) (blacks->method->head (blacks)))->id;
	
	/* Now we create the gray and white graph */
	work_on_graph (g, h1, h1p, blacklist, blacklength, WHITE);
	
	/* Now we create the gray and white graph */
	work_on_graph (g, h2, h2p, blacklist, blacklength, BLACK);


	/* We have h1, h2: a di-graph, with h1p, h2p and
	 * oppositely directioned graph. So, we decide
	 * on the colouring */
	
	edo_assignments (g, h1, h1p, h2, h2p, blacklist, blacklength);

	/* Remeber to deallocate */
	delete_graph (h1);
	delete_graph (h2);
	delete_graph (h1p);
	delete_graph (h2p);

	free (blacklist);
	
	return;
}

__inline__ static void
edo_assignments (MonoGraph *g, MonoGraph *h1, MonoGraph *h1p,
		MonoGraph *h2, MonoGraph *h2p, unsigned int *blacklist,
		unsigned int blacklength)
{
	unsigned int	count;
	unsigned int	i, u, w, z;
	bool		*done;
	List		*x [2];
	MonoGraph	*h [2];
	MonoGraph	*hp [2];

	/* We colour  x [0] black, and x [1] white */
	x [0] = create_list ();
	x [1] = create_list ();

	h [0] = h1, h [1] = h2;
	hp [0] = h1p, hp [1] = h2p;

	/* This is to indicate what is done and what is not.
	 * It's much quicker than using a list. */
	done = (bool *) malloc (sizeof (bool) * blacklength);
	memset (done, (int) false, sizeof (bool) * blacklength);

	/* u and w are always going to be the 
	 * vertex numbers of the vertices in h [i] graphs,
	 * and never in g */
	u = get_undone_vertex (done, blacklength);
	x [0]->method->append (x [0], g->vlist [ blacklist [u] ]);
	count = blacklength - 1;
	i = 1;
	
	while ( count ) {
		if ( (w = unplaced_successor (h [i - 1], u, done) ) < blacklength ) { 
			i = 3 - i;
			done [w] = true;
			/* This is the opposite colour now, because we've
			 * done a i = 3 - i */
			x [i - 1]->method->append (x [i - 1], g->vlist [ blacklist [w] ]);
			count--;
			
			/* We try this */ 
			if ( ( z = unplaced_successor ( hp [3 - i - 1], u, done) ) < blacklength ) { 
				done [z] = true;
	
				/* Opposite colour, we've already flipped i */
				x [i - 1]->method->append (x [i - 1], g->vlist [ blacklist [z] ]);
				count--;
	
				u = z;
			}
			else 
				u = w;
		
		}
		else if ( ( w = unplaced_successor ( hp [i - 1], u, done) ) < blacklength ) { 
			i = 3 - i;
			done [w] = true;

			/* Opposite colour, we've already flipped i */
			x [i - 1]->method->append (x [i - 1], g->vlist [ blacklist [w] ]);
			count--;
			 
			u = w;
		}
		else { 
			u = get_undone_vertex (done, blacklength); 
			x [0]->method->append (x [0], g->vlist [ blacklist [u] ]);
			count--;
			
			i = 1;
		}
	}

	/* Now we go and do the colourings. We need not worry
	 * about x [1] because we expect all the nodes to be black anyway. */
	colour_white (x [0]);

	delete_list (x [0]);
	delete_list (x [1]);

	free (done);

	return;
}
