/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    exception.c  - part of libexception.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <exception/exception.h>

/* Declarations, internal routines */
static EXAction default_handler (Exception *exception);

/* Declarations, Methods */
static EXOptions _get_options (Exception *exception);
static EXType _get_type (Exception *exception);
static EXPersonalType _get_ptype (Exception *exception);
static char *_get_string (Exception *exception);


/* Internal Globals */
static EXAction 	(*handler) (Exception *exception) = &default_handler; /* Default */
static EXAction 	action;
static Exception 	*cur_exception = NULL; /* Current exception */
static char 		*message = "An Exception has occured in your program"; /* Default message, ie bug message */
static FILE 		*nonstderr = NULL; /* Default stream to print output to */
static unsigned int 	_try = 0; /* Is there anything being tried? */

static ExceptionOps def_ops = {
	get_options:	_get_options,
	get_type: 	_get_type,
	get_ptype: 	_get_ptype,
	get_string: 	_get_string
};

/* Externals */
extern Exception 
*exception_raise (EXType type, EXPersonalType catch_type, EXOptions options, char *message, ...)
{
	int		vsn_return;
	unsigned int	size;
	char		*msg_copy = NULL;
	va_list		args;

	if ( !nonstderr )
		nonstderr = stderr;
	
	/* Shit, we've already got an exception */
	if ( cur_exception )
		action = handler (cur_exception);

	/* Clear current standing exceptions */
	exception_clear ();
	
	cur_exception = (Exception *) malloc (sizeof (Exception));
	
	/* Make sure exception handler doesn't run out of memmory! */
	if ( !cur_exception )
		OUT_OF_MEMMORY ();

	cur_exception->type = type;
	cur_exception->catch_type = strdup (catch_type);
	cur_exception->options = options;
	cur_exception->ops = &def_ops;

	/* No message */
	if ( !message ) {
		cur_exception->message = (char *) malloc (sizeof (char));

		if ( !cur_exception->message )
			OUT_OF_MEMMORY ();
		
		strcpy (cur_exception->message, " ");
		return cur_exception;
	}

	/* Message */
	size = TRUNCATE_LIMIT;

	while (1) {
		/* Space for message */
		msg_copy = (char *) malloc (sizeof (char) * size);
		
		if ( !msg_copy )
			OUT_OF_MEMMORY ();
		
		/* Attempt to print message into string */
		va_start (args, message);
		vsn_return = vsnprintf (msg_copy, size, message, args);
		va_end (args);

		/* It was big enough... */
		if ( vsn_return > -1 && vsn_return < size )
			break;

		/* Try again */
		free (msg_copy);
		size *= 2;
	}

	/* Conserve space */
	realloc (msg_copy, sizeof (char) * strlen (msg_copy));

	cur_exception->message = msg_copy;

	return cur_exception;
}

extern void
exception_clear (void)
{
	if ( !cur_exception ) 
		return;

	if ( cur_exception->catch_type )
		free (cur_exception->catch_type);
	
	free (cur_exception->message);
	free (cur_exception);

	cur_exception = NULL;
	/* It's already been handled ? */
	_try = 0;

	return;
}

extern void
exception_try (void)
{
	_try = 1;

	return;
}

extern Exception 
*exception_catch (EXPersonalType catch_type)
{
	/* Let's hope this doesn't happen! */
	if ( !cur_exception )
		return NULL;
	
	if ( !_try )
		fprintf (stderr, "Cannot exception_catch(), without exception_try()ing!");
	
	if ( cur_exception->catch_type )
		if ( strcmp (cur_exception->catch_type, catch_type ) )
			return NULL;

	/* We don't wanna do this here, because it could get exception_rethrow()n.
	 * The surest way is to stick it in exception_clear(), because then we
	 * know that the exception has been handled properly */

	/* _try = 0; */
	
	return cur_exception;
}

extern void
exception_handle (void)
{
	/* No Exceptions in Queue */
	if ( !cur_exception ) 
		return;
	
	handler (cur_exception);
	exception_clear ();

	return;
}
		
extern void
exception_rethrow (Exception *exception)
{
	if ( cur_exception )
		handler ( cur_exception );

	cur_exception = exception;

	return;
}

extern void
exception_set_handler (EXAction (*exception_handler) (Exception *exception))
{
	if ( exception_handler )
		handler = exception_handler;

	return;
}

extern void
exception_reset_handler (void)
{
	handler = default_handler;

	return;
}

extern void
exception_set_message (char *default_message)
{
	if ( default_message )
		message = strdup (default_message);

	return;
}

extern void
exception_set_stream (FILE *stream)
{
	if ( !stream ) 
		return;

	nonstderr = stream;

	return;
}

extern int
exception_type_compare (EXPersonalType type1, EXPersonalType type2)
{
	return !strcmp (type1, type2);
}

/* Methods */
static EXOptions 
_get_options (Exception *exception)
{
	if ( !exception ) 
		return 0;

	return exception->options;
}

static EXType
_get_type (Exception *exception)
{
	if ( !exception ) 
		return 0;

	return exception->type;
}

static EXPersonalType
_get_ptype (Exception *exception)
{
	if ( !exception )
		return NULL;

	return exception->catch_type;
}

static char
*_get_string (Exception *exception)
{
	if ( !exception )
		return NULL;

	return exception->message;
}

/* The Default Handler */
static EXAction
default_handler (Exception *exception)
{
	EXOptions 	options = exception->ops->get_options (exception);
	EXType		type = exception->ops->get_type (exception);

	if ( type == BUG )
		fprintf (nonstderr, "%s!\n", message );

	/* Print message to nonstderr */
	fprintf ( nonstderr, "%s", exception->ops->get_string (exception) );
	fprintf ( nonstderr, "\n");

	/* Make sure message is delivered */
	fflush (nonstderr);
	
	/* Don't panic if message is warning or information */
	if ( type == FATAL || type == ERROR ) 
		abort ();

	/* Well, it has been handled... hasn't it? */
	exception_clear ();

	return options;
}
