/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    generator.c  - part of libgenerator.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <generator/generator.h>

static void initialise (void);
static void make_graph (void);

__inline__ static void init_random (void);
__inline__ static void place (unsigned int i, unsigned int j);
__inline__ static void update_list (unsigned int index);

/* We make this = n(n-1)/2. This is a
 * much more efficient solution than using
 * a linked list */
static Edge		*edge_list;
static unsigned int	edge_count;
static unsigned int	tot_count;
static unsigned int 	big_r;
static MonoGraph	*graph;
static long int		seed;
static bool		noseed = false;

extern MonoGraph
*generate_graph (unsigned int vertices)
{
	if ( vertices < DEGREE + 1 ) {
		exception_raise (FATAL,
				"CannotGenerateGraph",
				NONE,
				"Graph Generator called with %d vertice",
				vertices);

		exception_handle ();
	}

	graph = create_graph (vertices);

	do {
	
		/* We initialise our parameters */
		initialise ();
		
		/* We attempt to create a graph */
		make_graph ();
	
		free (edge_list);

		/* Terminating condition */
		if ( tot_count - edge_count != 2 * graph->vertices ) {
			graph->method->reset (graph);
			/* In case we've fixed our seed... if we did
			 * then we'll be looping forever! */
			noseed = false;
		}
		else break;

	} while (1);
	
	return graph;
}

extern MonoGraph
*generate_graph_seed (unsigned int vertices, long int s)
{
	seed = s;

	noseed = true;

	return generate_graph (vertices);
}


static void
initialise (void)
{
	unsigned int	i, j;

	/* We initialise our random number generator */
	init_random ();

	tot_count =  graph->vertices * (graph->vertices - 1); 
	tot_count /= 2;

	edge_list = (Edge *) malloc (sizeof (Edge) * tot_count);

	edge_count = 0;

	MALLOC_ASSERT (edge_list);

	/* This is true, because if you look in make_graph,
	 * we omit updating big_r to begin with. This
	 * statement is true because we run the general case
	 * of the altorithm with an edge pair randomly assigned.
	 * So, we need to -4^2 + 3^2. */
	big_r = 16 * tot_count;

	/* Generate all possible edge pairs. 
	 * This sorts the list. This is really important */
	for (i = 0; i < graph->vertices; i++ )
		for (j = i + 1; j < graph->vertices; j++ )
			place (i,j);

	return;
}

static void 
make_graph (void)
{
	double		ran;
	unsigned int	a,b, mid;

	/* We can optimise this search solution
	 * further by computing a, b values
	 * more cleverly. Obviously, such a
	 * heuristic needs to be admissible, other wise,
	 * we're not going to find a solution. */
	
	while ( big_r ) {
		ran = drand48 () * big_r;
		
		/* We reduce our search time */
		a = 0;
		b = edge_count;
		mid = (a+b)/2;
		while (1) {
			 if ( ran <= edge_list [mid].interval )
				 b= mid;
			 else
				 a = mid;

			 if ( a == (b - 1) || a == b ) {
				 if ( ran <= edge_list [a].interval )
					 b = a;

				 break;
			 }
			 
			 mid = (a+b)/2;
		}

		/* Now a = b contains our element */

		graph->method->uconnect (graph,
				edge_list [b].u,
				edge_list [b].v);

		update_list (b);
	}

	/* The safe way */
#if 0
	while ( big_r ) {
		ran = drand48 () * big_r;

		for ( a = 0; a < edge_count; a++ )
			if ( ran <= edge_list [a].interval )
				break;

		graph->method->uconnect (graph,
				edge_list [a].u,
				edge_list [a].v);

		update_list (a);

	}
#endif 

	return;
}

__inline__ static 
void init_random ()
{
	int			efd;
	long int		myseed;

	efd = open ("/dev/random", O_RDONLY);

	if ( efd == -1 )
		exception_raise (FATAL, "CouldNotOpenEntropy", NONE,
				"Could not open /dev/random");

	/* Read sizeof (long int) * 1 amount of data */
	read (efd, &myseed, sizeof (long int));
	
	if ( !noseed )
		seed = myseed;
	
	srand48 (seed);

#ifdef __DEBUG__
	printf ("seed = %ld\n", seed);
#endif 

	close (efd);

	return;
}

/* The whole idea behind place is
 * so that we can uniformly space out
 * our edges in the list */
__inline__ static void
place (unsigned int i, unsigned int j)
{
	
	edge_list [edge_count].u = i; 
	edge_list [edge_count].v = j;

	edge_list [edge_count].r = DEGREE * DEGREE;
	
	edge_list [edge_count].interval = (edge_count + 1) * edge_list [edge_count].r;
	edge_count++;
	
	return;
}

__inline__ static void
update_list (unsigned int index)
{

	Edge	e;
#ifdef __DEBUG__
	printf ("edge (%d, %d)\n", 
			edge_list [index].u,
			edge_list [index].v);
#endif 
	/* We've removed this edge from our list now */
	big_r -= edge_list [index].r; 

	/* Keep a copy of our edge */
	memcpy (&e, edge_list + index, sizeof (Edge));
	memcpy (edge_list + index,  edge_list + (edge_count - 1), sizeof (Edge));

	/* We have to update our interval and r values now! */

	/* Base case */
	if ( edge_list [0].u == e.u || 
			edge_list [0].u == e.v ) {
		big_r += graph->vlist [edge_list [0].v]->neighbours - 4;
		edge_list [0].r += graph->vlist [edge_list [0].v]->neighbours - 4;
	}
	else if ( edge_list [0].v == e.u ||
			edge_list [0].v == e.v ) {
		big_r += graph->vlist [edge_list [0].u]->neighbours - 4;
		edge_list [0].r += graph->vlist [edge_list [0].u]->neighbours - 4;
	}
		
	edge_list [0].interval = edge_list [0].r;
		
	
	/* Ok, we update for general case */
	for ( index = 1, edge_count--; index < edge_count; index++ ) {
		if ( edge_list [index].u == e.u || 
				edge_list [index].u == e.v ) {
			big_r += graph->vlist [edge_list [index].v]->neighbours - 4;
			edge_list [index].r += graph->vlist [edge_list [index].v]->neighbours - 4;
		}
		else if ( edge_list [index].v == e.u ||
				edge_list [index].v == e.v ) {
			big_r += graph->vlist [edge_list [index].u]->neighbours - 4;
			edge_list [index].r += graph->vlist [edge_list [index].u]->neighbours - 4;
		}
		
		edge_list [index].interval = edge_list [index - 1].interval + edge_list [index].r;
	}
	
	return;
	
}
