/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    chromon.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <monograph/chromon.h>

/* Method Prototypes */
static void _discover (ChromonList *c, MonoGraph *g);
static void _rediscover (ChromonList *c, List *dlist); 

/* Internal Functions */
__inline__ static void initialise_discover_metadata (ChromonList *c, MonoGraph *g);
__inline__ static void discover_on_vertice (ChromonList *c, Node *n);
__inline__ static void delete_chromons (List *chromon_list);
static void redis_func (Node *n, ChromonList *c);

/* Method Skeleton structure */
static ChromonOps	operations = {
	discover:		&_discover,
	rediscover:		&_rediscover,
};

/* Constructor and Destructor */

extern ChromonList
*create_chromonlist (void)
{
	ChromonList	*c;

	c = (ChromonList *) malloc (sizeof (ChromonList));

	MALLOC_ASSERT (c);

	memset (c, (int) NULL, sizeof (ChromonList));


	/* Create access to the methods */
	c->method = &operations;

	return c;
}

extern void
delete_chromonlist (ChromonList *c)
{
	
	DEBUG_ASSERT ( c != NULL );

	if ( c->vertice )
		free (c->vertice);


	delete_chromons (c->black);
	delete_chromons (c->white);
	
	delete_list (c->black);
	delete_list (c->white);

	free (c);

	return;
}

/* Methods */
static void
_discover (ChromonList *c, MonoGraph *g)
{
	unsigned int	i;
	
	DEBUG_ASSERT ( c != NULL );

	initialise_discover_metadata (c, g);

	
	/* Now we go and detect the length of our chromon runs 
	 * for each vertice */
	for ( i = 0; i < g->vertices; i++ )
		/* Only work on vertice if it's not
		 * already allocated */
		if ( !c->vertice [i].list )
			discover_on_vertice (c, c->graph->vlist [i]);

	return;
}

static void
_rediscover (ChromonList *c, List *dlist)
{
	Node	*n;
	List	*ndlist;
	List	*chromon;

	DEBUG_ASSERT ( c != NULL );

	if ( !c->graph )
		return;

	ndlist = create_list ();

	while ( dlist->length ) {
		n = dlist->method->head (dlist);

		/* If we have a node, then we have a chromon
		 * corresponding to that node... */
		chromon = c->vertice [n->id].list;

		/* If this is not the case, then we've already
		 * processed another node from this chromon */
		if ( chromon ) {
			/* We need to rebuild all these chromons again. We
			 * need to look at the opposite colour here because
			 * we've flipped n. So, it used to belong to BLACK,
			 * if it is now white. */
			switch (n->colour) {
				case WHITE: 	if ( !c->black->method->delete (c->black, chromon) ) 
							c->white->method->delete (c->white, chromon);
						break;
						
				case BLACK: 	if ( !c->white->method->delete (c->white, chromon) )
							c->black->method->delete (c->black, chromon);
						break;
			}
	
			/* We go and set each node in this chromon to
			 * c->vlist [n->id].list = NULL; */
			chromon->method->map (chromon, &redis_func, c);
			ndlist->method->merge (ndlist, chromon);
		}
	}

	while ( ndlist->length ) {
		n = ndlist->method->head (ndlist);
		
		if ( !c->vertice [n->id].list )
			discover_on_vertice (c, n);
	}

	return;  
}

/* Internal functions */

__inline__ static void
initialise_discover_metadata (ChromonList *c, MonoGraph *g)
{
	DEBUG_ASSERT (g);

	c->graph = g;
	c->black = create_list ();
	c->white = create_list ();

	c->vertice = (ChromonV *) malloc (sizeof (ChromonV) * g->vertices);

	MALLOC_ASSERT (c->vertice);

	memset (c->vertice, (int) NULL, sizeof (ChromonV) * g->vertices);

	return;
	
}

__inline__ static void
discover_on_vertice (ChromonList *c, Node *n)
{
	List		*process_list; /* List to know which to process */
	List		*chromon;      /* List of our chromon */
	unsigned int	i;
	
	/* We've already got this in a chromon. So we don't bother */
	if ( c->vertice [n->id].list )
		return;

	process_list = create_list ();
	chromon = create_list ();

	/* We add our vertice for processing */
	process_list->method->append (process_list, n);

	/* We go off and build our chromon */
	while ( process_list->length ) {
		n = process_list->method->head (process_list);

		
		/* We haven't already allocated this vertice */
		if ( !c->vertice [n->id].list ) { 
			for ( i = 0; i < n->neighbours; i++ )
				/* We need to go off and process our co-chromatic
				 * neighbours */
				if ( n->neighbour [i]->colour == n->colour )
					process_list->method->append (process_list, n->neighbour [i]);

			chromon->method->append (chromon, n);

			c->vertice [n->id].list = chromon;
		}
	}

	/* We have no use for this */
	delete_list (process_list);

	/* We want to add the chromon to the appropriate
	 * chromon partition */
	switch (n->colour) {
		case BLACK:	c->black->method->append (c->black, chromon);
				break;
		case WHITE: 	c->white->method->append (c->white, chromon);
				break;
	}


	return;
}

__inline__ static void
delete_chromons (List *chromon_list)
{
	List		*chromon;
	
	while ( chromon_list->length ) {
		chromon = chromon_list->method->head (chromon_list);

		delete_list (chromon);
	}
	
	return; 
}

static void
redis_func (Node *n, ChromonList *c)
{
	c->vertice [n->id].list = NULL;

	return;
}
