/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    chromon.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <monograph/graph.h>

/* Methods */
static Node **_get_nodes (MonoGraph *g);
static Node *_get_node (MonoGraph *g, unsigned int node);
static unsigned int _get_size (MonoGraph *g);
static void _connect (MonoGraph *g, unsigned int vertice, unsigned int to);
static void _uconnect (MonoGraph *g, unsigned int vertice, unsigned int to);
static void _disconnect (MonoGraph *g, unsigned int vertice, unsigned int to);
static void _reset (MonoGraph *g);
static MonoGraph *_duplicate (MonoGraph *g);

/* Operations Structure */

static MonoGraphOps	operations = {
	get_nodes: 	&_get_nodes,
	get_node: 	&_get_node,
	get_size:	&_get_size,
	connect:	&_connect,
	uconnect:	&_uconnect,
	disconnect:	&_disconnect,
	reset:		&_reset,
	duplicate:	&_duplicate
};

/* Constructors and Destructor */
extern MonoGraph
*create_graph (unsigned int vertices)
{
	MonoGraph 	*g;
	int		i;
	
	/* We get a graph */
	g = (MonoGraph *) malloc (sizeof (MonoGraph));
	MALLOC_ASSERT (g);

	/* We can't have zero vertices or less */
	METHOD_ASSERT ( vertices != 0 );

	/* We need to init the graph, because we perform
	 * NULL checks */
	memset (g, (int) NULL, sizeof (MonoGraph));
	
	g->vertices = vertices;
	g->method = &operations;

	/* We get the node pointer vector */
	g->vlist = (Node **) malloc (sizeof (Node *) * vertices);
	MALLOC_ASSERT (g->vlist);

	/* Get all our nodes */
	for (i = 0; i < vertices; i++)
		g->vlist [i] = create_node (i);

	/* TODO: Chromon List */

	return g; 
}

extern void
delete_graph (MonoGraph *g)
{
	int 	i;
	
	DEBUG_ASSERT ( g != NULL );

	/* Get rid of the allocation for each node */
	for ( i = 0; i < g->vertices; i++ ) {
		delete_node (g->vlist [i]);
		g->vlist [i] = NULL;
	}

	/* Get rid of the allocation of node vector */
	free (g->vlist);

	return;
}

/* Methods */
static Node
**_get_nodes (MonoGraph *g)
{
	DEBUG_ASSERT ( g != NULL );

	return g->vlist;
}

static Node
*_get_node (MonoGraph *g, unsigned int node)
{
	DEBUG_ASSERT ( g != NULL );
	DEBUG_ASSERT ( node < g->vertices );

	return g->vlist [node];
}

static unsigned int
_get_size (MonoGraph *g)
{
	DEBUG_ASSERT ( g!= NULL );

	return g->vertices;
}

static void
_connect (MonoGraph *g, unsigned int vertice, unsigned int to)
{
	DEBUG_ASSERT ( g != NULL );
	DEBUG_ASSERT ( vertice < g->vertices );
	DEBUG_ASSERT ( to < g->vertices );
	
	/* We don't allow circles in our graphs */
	DEBUG_ASSERT ( vertice != to );

	/* Our MonoGraph is undirected. So we need to add link both ways */
	g->vlist [vertice]->method->add_neighbour (g->vlist [vertice], g->vlist [to]);

	return;
}

static void
_uconnect (MonoGraph *g, unsigned int vertice, unsigned int to)
{
	DEBUG_ASSERT ( g != NULL );
	DEBUG_ASSERT ( vertice < g->vertices );
	DEBUG_ASSERT ( to < g->vertices );
	
	/* We don't allow circles in our graphs */
	DEBUG_ASSERT ( vertice != to );

	/* Our MonoGraph is undirected. So we need to add link both ways */
	g->vlist [vertice]->method->add_neighbour (g->vlist [vertice], g->vlist [to]);
	g->vlist [to]->method->add_neighbour (g->vlist [to], g->vlist [vertice]);

	return;
	
}

static void
_disconnect (MonoGraph *g, unsigned int vertice, unsigned int to)
{
	DEBUG_ASSERT ( g != NULL );
	DEBUG_ASSERT ( vertice < g->vertices );
	DEBUG_ASSERT ( to < g->vertices );

	/* We can't have this: */
	DEBUG_ASSERT ( vertice != to );

	/* The neighbour doesn't neccessarily need to exist. */
	g->vlist [vertice]->method->del_neighbour (g->vlist [vertice], g->vlist [to]);

	return;
}

static void
_reset (MonoGraph *g)
{
	unsigned int	i;
	
	for (i = 0; i < g->vertices; i++ )
		g->vlist [i]->method->reset (g->vlist [i]);

	return;
}

static MonoGraph
*_duplicate (MonoGraph *g)
{
	MonoGraph 	*new;
	unsigned int	i, j;

	DEBUG_ASSERT (g != NULL);
	
	if ( !g->vertices )
		return NULL;
	
	new = create_graph (g->vertices);

	/* Go through all vertices and neighbours, and
	 * add them appropriately to the new graph */
	for ( i = 0; i < g->vertices; i++ ) { 
		new->vlist [i]->neighbours = g->vlist [i]->neighbours;

		for ( j = 0; j < g->vlist [i]->neighbours; j++ ) 
			new->vlist [i]->neighbour [j] = 
				new->vlist [ g->vlist [i]->neighbour [j]->id ];
	}
			
	return new;
}
