/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    chromon.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <monograph/list.h>

/* Method Prototypes */
static void *_head (List *l);
static void *_ndhead (List *l);
static void *_next (List *l);
static void *_prev (List *l);
static void _map (List *l, void (*mapfun) (void *data, void *param), void *param);

static ListNode *_cons (List *l, void *data);
static ListNode *_append (List *l, void *data);
static List *_merge (List *l, List *m);

static void *_delete (List *l, void *data);
static unsigned int _length (List *l);
static unsigned int _nlength (ListNode *n);
static void _reset (List *l);

/* Operations structure */
static ListOps		operations = {
	head: 		&_head,
	ndhead: 	&_ndhead,
	next:		&_next,
	prev:		&_prev,
	map:		&_map,

	cons:		&_cons,
	append: 	&_append,
	merge:		&_merge,

	delete:		&_delete,
	
	length:		&_length,
	nlength: 	&_nlength,
	reset: 		&_reset
};

/* Construct and Destruct */
extern List 
*create_list (void)
{
	List	*l;

	l = (List *) malloc (sizeof (List));

	MALLOC_ASSERT (l);
	
	memset (l, (int) NULL, sizeof (List));

	l->method = &operations; 

	return l;
}

extern void
delete_list (List *l)
{
	ListNode	*cur = l->root;
	ListNode	*next;

	while ( cur ) {
		next = cur->next;

		free (cur);

		cur = next;
	}
	
	free (l);
}

/* Methods */
static void
*_head (List *l)
{
	void		*data;
	ListNode	*cur;

	DEBUG_ASSERT ( l != NULL );

	/* We don't have a head! */ 
	if ( ! l->root )
		return NULL;

	cur = l->root;
	data = cur->data;
	l->root = cur->next;

	if ( l->last == cur ) 
		l->last = NULL;

	/* We need to step length down */
	l->length--;

	free (cur); 

	return data;
}

static void
*_ndhead (List *l)
{
	DEBUG_ASSERT ( l != NULL );

	return l->root->data;
}

static void
*_next (List *l)
{
	DEBUG_ASSERT ( l != NULL );

	if ( !l->length ) 
		return NULL;

	if ( !l->cur ) {
		l->cur = l->root;
		return l->cur->data;
	}
	
	l->cur = l->cur->next;

	if ( l->cur ) 
		return l->cur->data;

	/* We're at the end */
	return NULL;
}

static void 
*_prev (List *l)
{
	DEBUG_ASSERT ( l != NULL );

	if ( !l->length )
		return NULL;
	
	if ( !l->cur ) {
		l->cur = l->last;
		return l->cur->data;
	}
	
	l->cur = l->cur->prev;

	if ( l->cur )
		return l->cur->data;

	return NULL;
}

#define ADD_PREAMBLE(node,data) 				\
	{							\
		n = (ListNode *) malloc (sizeof (ListNode));	\
								\
		MALLOC_ASSERT (n);				\
								\
		/* Copy Data */					\
		n->data = data;					\
		n->next = NULL;					\
								\
		/* Increment length */ 				\
		l->length++;					\
								\
		if ( !l->root ) {				\
			l->last = l->root = n;			\
			n->prev = NULL;				\
			return n;				\
		}						\
	}

		

static ListNode
*_cons (List *l, void *data)
{
	ListNode	*n;

	DEBUG_ASSERT ( l != NULL ); 

	ADD_PREAMBLE (n, data);

	/* We have O(1) cons */
	n->next = l->root;
	l->root->prev = n;
	n->prev = NULL;

	/* The root node now becomes n */
	l->root = n;

	return n;
}

static ListNode
*_append (List *l, void *data)
{
	ListNode	*n;

	DEBUG_ASSERT ( l != NULL );

	ADD_PREAMBLE (n, data);
	
	/* Yep, we have O(1) append. This is good! */
	l->last->next = n;
	n->prev = l->last;
	n->next = NULL;
	
	/* Set the last pointer to n */ 
	l->last = n;

	return n;
}

static List
*_merge (List *l, List *m)
{
	DEBUG_ASSERT ( l != NULL );
	DEBUG_ASSERT ( m != NULL );

	if ( !l->length ) { 
		memcpy (l, m, sizeof (List));
		free (m);
		return l;
	}
	
	l->length += m->length;

	/* Make appropriate pointer links */
	l->last->next = m->root;
	m->root->prev = l->last;

	l->last = m->last;

	/* Declare zero length for m, so we don't
	 * remove any of the nodes */
	m->root = m->last = NULL;
	m->length = 0;

	free (m);

	return l;
}

static void
_map (List *l, void (*mapfunc) (void *data, void *param), void *param)
{
	ListNode	*cur;

	cur = l->root;

	while ( cur ) {
		mapfunc (cur->data, param);

		cur = cur->next;
	}

	return;
}

static void
*_delete (List *l, void *data)
{
	ListNode	*cur;

	DEBUG_ASSERT ( l != NULL );
	
	/* Base case, we have one element! */
	if ( l->length == 1 ) {
		if ( l->root->data != data )
			return NULL;

		l->root = NULL;
		l->last = NULL;

		free (l->cur);
		
		l->cur = NULL;
		l->length = 0;

		return data;
	}
		
	
	/* Easy way to del when we have
	 * cur properly aligned */
	if ( l->cur )
		if ( l->cur->data == data ) { 
			cur = l->cur;
			goto add;
		}
	
	/* We start with first element */
	cur = l->root;

	while ( cur ) { 
		/* We have a match */ 
add: 		if ( cur->data == data ) {
			/* Set the pointers right */
			if ( cur == l->root ) {
				cur->next->prev  = NULL;
				l->root = cur->next;
			}
			else  
				cur->prev->next = cur->next;
			
			/* Last element */
			if ( cur == l->last ) {
				cur->prev->next = NULL;
				l->last = cur->prev;
			}
			else 
				cur->next->prev = cur->prev;

			/* This is rather arbitary */
			if ( cur == l->cur )
				l->cur = cur->prev;

			free (cur);
			
			l->length--;

			return data;
		}

		/* Step throught the list */
		cur = cur->next;
	}

	/* Data isn't contained in this list! */
	return NULL;
}

static unsigned int
_length (List *l)
{
	DEBUG_ASSERT ( l != NULL );

	return l->length; 
}

static unsigned int
_nlength (ListNode *n)
{
	unsigned int 	i;
	
	/* Nothing in node! */
	if ( !n )
		return 0;
	
	for ( i = 1; n != NULL; i++ )
		n = n->next;

	return i;
}

static void
_reset (List *l)
{
	l->cur = NULL;

	return;
}
