/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    node.c  - part of libmonograph.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <monograph/node.h>

/* Method Prototypes */
static unsigned int _get_id (Node *n);
static void _add_neighbour (Node *n, Node *neighbour);
static Node *_get_neighbour (Node *n, unsigned int neighbour);
static void _del_neighbour (Node *n, Node *neighbour);
static unsigned int _neighbour_count (Node *n);
static Colour _get_colour (Node *n);
static void _set_colour (Node *n, Colour c);
static void _invert (Node *n);
static void _reset (Node *n);


/* Operations Structure */
static NodeOps 		operations = {
	get_id:			&_get_id,
	add_neighbour:  	&_add_neighbour,
	get_neighbour: 		&_get_neighbour,
	del_neighbour:		&_del_neighbour,
	neighbour_count:	&_neighbour_count,
	get_colour:		&_get_colour,
	set_colour: 		&_set_colour,
	invert:			&_invert,
	reset:			&_reset
};

/* Construct & Destruct */

extern Node
*create_node (unsigned int id)
{
	Node	*n;

	n = (Node *) malloc (sizeof (Node));

	MALLOC_ASSERT (n);

	memset (n->neighbour, (int) NULL, sizeof (Node *) * DEGREE);
	n->method = &operations;

	n->id = id;
	n->neighbours = 0;
	n->colour = BLACK;

	return n;
}

extern void
delete_node (Node *n)
{
	METHOD_ASSERT ( n != NULL );

	free (n);

	return;
}

/* Methods */
static unsigned int
_get_id (Node *n)
{
	DEBUG_ASSERT ( n != NULL );
	
	return n->id;
}

static void 
_add_neighbour (Node *n, Node *neighbour)
{
	unsigned int	i;
	
	DEBUG_ASSERT ( n != NULL );
	DEBUG_ASSERT ( neighbour != NULL );
	
	if ( n->neighbours == DEGREE ) {
		exception_raise (WARNING, "CannotAddNeighbour", NONE,
				"Failed to add neighbour!");
		return;
	}

	/* We've already added the neighbour! */
	for ( i = 0; i < n->neighbours; i++ )
		if ( n->neighbour [i] == neighbour ) {
			exception_raise (WARNING, "CannotAddNeighbour", NONE,
					"Neighbour exists!");
			return;
		}
	
	/* Clean add! */
	n->neighbour [n->neighbours++] = neighbour;

	return;
}

static Node
*_get_neighbour (Node *n, unsigned int neighbour)
{
	DEBUG_ASSERT ( neighbour < DEGREE );

	return n->neighbour [neighbour]; /* Could be NULL */
}

static void
_del_neighbour (Node *n, Node *neighbour)
{
	unsigned int i;

	DEBUG_ASSERT ( n != NULL );
	DEBUG_ASSERT ( neighbour != NULL );

	for ( i = 0; i < n->neighbours; i++ )
		if ( n->neighbour [i] == neighbour ) {
			n->neighbour [i] = NULL;
			n->neighbour [i] = n->neighbour [--(n->neighbours)];
			n->neighbour [n->neighbours] = NULL;
		}

	return;
}

static unsigned int
_neighbour_count (Node *n)
{
	DEBUG_ASSERT ( n != NULL );
	
	return n->neighbours;
}

static Colour
_get_colour (Node *n)
{
	DEBUG_ASSERT ( n != NULL );
	
	return n->colour;
}

static void
_set_colour (Node *n, Colour c)
{
	DEBUG_ASSERT ( n != NULL );

	n->colour = c;
}

static void
_invert (Node *n)
{
	DEBUG_ASSERT ( n != NULL );

	switch (n->colour) {
		case BLACK: 	n->colour = WHITE; break;
		case WHITE: 	n->colour = BLACK; break;
	}

	return;
}

static void
_reset (Node *n)
{
	DEBUG_ASSERT ( n != NULL );

	memset (n->neighbour, (int) NULL, sizeof (Node *) * DEGREE);
	n->neighbours = 0;
	n->colour = BLACK;

	return;
}
