/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    graphwrite.c - part of graphwrite.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h> 
#include <stdlib.h>

#include <monograph/graph.h>
#include <generator/generator.h>

/* Prototypes */
static void parse_options (int argc, char **argv);
static void make_directed (void);
static void print_marker (void);
static void write_graph (void);
static void help (void);

/* Internal Globals */
static FILE		*ostream;
static unsigned int	graphs;
static unsigned int	vertices;
static MonoGraph	*graph;


int
main (int argc, char **argv)
{
	unsigned int	i;
	
	parse_options (argc, argv);

	for ( i = 0; i < graphs; i++ ) {
		/* Generate graph */ 
		graph = generate_graph (vertices);

		/* Make directed! */
		make_directed ();

		/* Write the graph to outstream */
		write_graph ();
		
		/* We print out a marker to separate
		 * each graph. */
		print_marker ();

		delete_graph (graph);		
	}

	if ( ostream != stdout )
		fclose (ostream);

	return 0;
}

static void
parse_options (int argc, char **argv)
{
	/* Keep it simple for the time being.
	 * We expect 
	 * 	(1) argv [1] = # of vertices
	 * 	(2) argv [2] = # of graphs
	 * 	(3) argv [3] = optional output file */

	if ( argc < 3 ) {
		help ();
		exit (1);
	}

	vertices = atoi (argv [1]);
	graphs = atoi (argv [2]);

	if ( !vertices || !graphs ) {
		fprintf (stderr, "(vertices, graphs) = (%d, %d)\n", vertices, graphs);
		exit (2);
	}

	/* Initialise to this */
	ostream = stdout;

	if ( argc < 4 )
		return;

	ostream = fopen (argv [3], "w");

	if ( !ostream ) {
		fprintf (stderr, "Your filname %s cannot be opened for writing.\n", argv [3]);
		ostream = stdout;
	}

	return;
}

static void
make_directed (void)
{
	unsigned int	i, j;
	
	/* We expect a graph */
	for ( i = 0; i < vertices; i++ )
		for ( j = 0; j < graph->vlist [i]->neighbours; j++ )
			graph->vlist [i]->neighbour [j]->method->del_neighbour (graph->vlist [i]->neighbour [j],
					graph->vlist [i]);

	return;
}

static void
print_marker (void)
{
	fprintf (ostream, ";\n");
}

static void
write_graph (void)
{
	unsigned int 	i, j;
	unsigned int	edge_count;

	/* We expect a graph  to be there!!! */

	for ( i = 0, edge_count = 0; i < vertices; i++ )
		for ( j = 0; j < graph->vlist [i]->neighbours; j++ ) { 
			fprintf (ostream, "(%d, %d)\n",
					i,
					graph->vlist [i]->neighbour [j]->id);
			
			edge_count++;
		}

	if ( edge_count != (vertices * 2) )
		fprintf (stderr, "edge_count != vertices\n");

	
	return;
}

static void
help (void)
{
	fprintf (stderr, "Copyright/Copyleft (c) 2004, Menaka Lashitha Bandara\n");
	fprintf (stderr, "This software is distributed under the GNU Public License version 2.\n");
	fprintf (stderr, "Please visit www.gnu.org for more information\n\n");
	
	fprintf (stderr, "Usage: graphwrite [vertices] [graphs] [[output_file]]\n");
	fprintf (stderr, "If no output file is given, the graph(s) will be written to stdout\n");
}
