/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    process.c - part of process.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "process.h"

/* Internal functions */
static void parse_arguments (int argc, char **argv);
static MonoGraph *get_from_file (void);
static MonoGraph *randomly_generate (void);

static void process (void);
static Data *run_algorithm (MonoGraph *master_graph, ChromonList *(*algorithm) (MonoGraph *graph));
__inline__ static unsigned int get_max_chromon (ChromonList *c);

static void reset_colour (MonoGraph *g);

static void report_results (List *list);
__inline__ static unsigned long int imax (unsigned long int a, unsigned long int b);
__inline__ static float f_max (float a, float b);
__inline__ static unsigned long int isquare (unsigned long int a);
__inline__ static float fsquare (float a);

static void clean_up (void); 

static void help (void);
static void version (void);

static char *license = 
" This program is free software; you can redistribute\n\
 it  and/or   modify  it  under  the  terms  of  the\n\
 GNU General Public License   as  published  by  the\n\
 Free Software Foundation;  either  version 2 of the\n\
 License, or  (at  your  option)  any later version.";


#ifdef __DEBUG__ 
/* Debug */
static void colour_graph (MonoGraph *g); 
#endif /* __DEBUG__ */


/* Internal stuff */ 
static FILE	 	*stream = NULL;		/* Input Stream */ 
static FILE		*ostream = NULL;	/* Output Stream */ 
static long int		vertices = 0;		/* For random */ 
static long int		max = 1;		/* Maximum number of runs */
static bool		usemax = false;
static MonoGraph *(*get_graph) (void) = NULL;	/* How do we get our next graph? */

/* This is our options */
const static struct option	long_opt [] = {
	{"help", 0, 0, 'h'},		/* Help */ 
	{"version", 0, 0, 'v'},		/* Version */ 
	{"input", 1, 0, 'i'},		/* Specify input file */ 
	{"output", 1, 0, 'o'},		/* Specify output file */ 
	{"max-limit", 1, 0, 'm'},	/* Maximum number to read/generate */
	{"size", 1, 0, 's'},		/* Number of vertices for graph */
	{NULL, 0, 0, 0},
};
const static char		*short_opt = "hvi:o:m:s:";

/* The algorithm structure */
static Algorithm	algorithm_list [] = {
	{&ef_algorithm, "EF", NULL},
	{&eef_algorithm, "EEF (Enhanced EF)",  NULL},
	{&e2ef_algorithm, "E2EF", NULL},
	{&hst_algorithm, "HST", NULL},
	{&ehst_algorithm, "EHST (LASHI)", NULL},
	{NULL, NULL, NULL}
};

int
main (int argc, char **argv)
{
	/* We set our default parameters */
	parse_arguments (argc, argv);

	/* Now we run our graphs */
	process ();

	/* Clean up */
	clean_up ();

	return 0;
}

static void
parse_arguments (int argc, char **argv)
{
	int	c;
	int 	opt_ind = 0;

	/* Our defaults */
	ostream = stdout;

	while ( (c = getopt_long (argc,
					argv,
					short_opt,
					long_opt,
					&opt_ind) ) != -1 ) {
		switch (c) {
			/* Help */
			case 'h': 	help ();
					break;

			/* Version */ 
			case 'v': 	version ();
					exit (0);
					break;
					
			/* Input file */
			case 'i':	stream = fopen (optarg, "r");
					if ( !stream ) { 
						exception_raise (FATAL,
								EXCEPTION_FATAL,
								NONE,
								"Failed to open %s for reading!",
								optarg);
						
						stream = stdin;
					}

					break;
					
			/* Output File */ 
			case 'o': 	if ( (ostream = fopen (optarg, "r")) ) { 
						exception_raise (FATAL,
								EXCEPTION_FATAL,
								NONE,
								"File %s exists! Not overwriting!",
								optarg);

						fclose (ostream); 
						break;
					}
					
					ostream = fopen (optarg, "w");
					
					if ( !ostream )
						exception_raise (FATAL,
								EXCEPTION_FATAL,
								NONE,
								"File %s cannot be open for writing!",
								optarg);

					break;
			
			/* Max limit */
			case 'm': 	max = atoi (optarg); /* We handle these errors later */
					usemax = true;
					break;
			
			/* Number of vertices */
			case 's': 	vertices = atoi (optarg);
					break;
					
			case '?': 	fprintf (stderr, "\n");
					help ();
					exit (1);
					break;
					
			default: 	exception_raise (FATAL,
							EXCEPTION_FATAL,
							NONE,
							"Something wicked happened! Aborting!");
		}
	}

	/* Unknown option! */
	if ( optind < argc )
		exception_raise (FATAL,
				EXCEPTION_FATAL,
				NONE,
				"Unknown option %s", argv [optind]);

	/* We handle exceptions so far, before we carry on */
	exception_handle ();
	
	/* By default, we do interactive */
	if ( !vertices && !stream ) {
		long int	v;
		printf ("Number of vertices to produce graph: ");
		scanf ("%d", &v); 

		if  ( v < 1 ) {
			exception_raise (FATAL,
					EXCEPTION_FATAL,
					NONE,
					"Cannot produce a graph of %d vertices", v);
		}

		vertices = (unsigned int) v;
	}
	
	/* Assume that we have some vertices, but havn't specified max,
	 * then we set max = 1. Note by this scheme, we give preference
	 * to reading graphs from a stream than randomly generating them */
	if ( vertices && !stream ) { 
		if ( max < 0 ) { 
			exception_raise (WARNING, 
					EXCEPTION_WARNING,
					NONE,
					"Attempting to create an infinite number of random graphs! Setting this limit to 1.");
			max = 1;
		}

		usemax = true;

		get_graph = &randomly_generate;
	}
	/* We read from the stream we've been given */
	else if ( !vertices ) { 
		get_graph = &get_from_file;
		
		if ( max < 0 ) {
			exception_raise (WARNING,
					EXCEPTION_WARNING,
					NONE,
					"Reading all the graphs from stream");
			usemax = false;
		}
	}

	/* We force this, so we don't need to proceed
	 * if we've got a waiting exception */
	exception_handle ();

	if ( ostream == stdout )
		fprintf (stderr, "Writing to stdout.\n");


	return;
}

static MonoGraph
*get_from_file (void)
{
	/* No more graphs in file */ 
	if ( feof (stream) )
		return NULL;

	return read_graph (stream);
}

static MonoGraph 
*randomly_generate (void)
{
	return generate_graph (vertices);
}

static void 
process (void)
{
	MonoGraph	*g;
	unsigned int	i, j;

	/* Create our lists */
	for ( i = 0; algorithm_list [i].algorithm ; i++ )
		algorithm_list [i].results = create_list ();
	
	/* Run the algorithms */
	for ( i = 0 ; usemax ? i < max: true; i++ ) {
		
		/* Read our graph */
		g = get_graph ();

		/* We can't go on! */
		if ( !g )
			break;

		/* Now we run each algorithm in the list */
		for ( j = 0; algorithm_list [j].algorithm ; j++ ) {
			reset_colour (g);
			algorithm_list [j].results->method->append (algorithm_list [j].results,
					run_algorithm (g, 
						algorithm_list [j].algorithm));
		}
		
		delete_graph (g);
	}
	
	/* Report results */
	for ( i = 0; algorithm_list [i].algorithm ; i++ ) {
		fprintf (ostream, "\n*** %s Algorithm ***\n",
				algorithm_list [i].name);
		report_results (algorithm_list [i].results);

		/* We also delete our results lists */
		delete_list (algorithm_list [i].results);
	};
	
	return;
}


static Data
*run_algorithm (MonoGraph *g, ChromonList *(*algorithm) (MonoGraph *graph) )
{
	ChromonList	*c;
	struct rusage	myuse [2];
	Data		*data;

	DEBUG_ASSERT ( g != NULL );

	/* We need to report our data */
	data = (Data *) malloc (sizeof (Data));
	MALLOC_ASSERT ( data != NULL );
	memset (data, (int) NULL, sizeof (Data));

	getrusage (RUSAGE_SELF, myuse);
	c = algorithm (g);
	getrusage (RUSAGE_SELF, myuse + 1);

	/* Reporting */
	data->time = 
		/* Seconds */
		((unsigned long long int)
		  (myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec )) * 1000000 +
		/* Microseconds */
		((unsigned long long int) 
		 (myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec));	
#if 0 
		/* Seconds */
		((unsigned long long int)
		  (myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec +
		   myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec )) * 1000000 +
		/* Microseconds */
		((unsigned long long int) 
		 (myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec +  
		  myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec));	
#endif

	data->max_chromon_size = get_max_chromon (c);
	data->average_chromon_size =
		((float) c->graph->vertices) / ((float) (c->black->length + c->white->length));

	delete_chromonlist (c);
	
	return data;
}
	


__inline__ static unsigned int
get_max_chromon (ChromonList *c)
{
	unsigned int	i;
	unsigned int 	largest;
	List		*l;
	
	c->black->method->reset (c->black);
	c->white->method->reset (c->white);
	
	/* Hunt for the largest chromon */
	for ( i = 0, largest = 0; i < c->black->length; i++ )
		if ( (l = c->black->method->next (c->black) ))
			if ( l->length > largest )
				largest = l->length;

	for ( i = 0; i < c->white->length; i++ )
		if ( (l = c->white->method->next (c->white) ))
			if ( l->length > largest )
				largest = l->length;

	return largest;
}

static void
reset_colour (MonoGraph *g)
{
	unsigned int	i;

	for (i = 0; i < g->vertices ; i++ )
		g->vlist [i]->colour = BLACK;

	return;
}


/* Report functions */
static void
report_results (List *list)
{
	unsigned int	i;
	
	/* We report these statistics last */
	unsigned long int 	average_max_chromon = 0;
	float			average_average_chromon = 0;
	unsigned long int	max_max_chromon = 0;
	float			max_average_chromon = 0;
	unsigned long int	average_time = 0;
	unsigned long int 	max_time = 0;

	/* Standard Deviations */
	unsigned long int	sd_max_chromon = 0; 
	float			sd_av_chromon = 0;
	unsigned long int	sd_time = 0;

	Data			*data;

	/* We go off and calculate all of the above */

	fprintf (ostream, "Time (us) || Max. Chr. || Av. Chr.\n");

	for ( i = 0; list->length ; i++ ) {
		data = list->method->head (list);
		
		/* We print each entry */
		fprintf (ostream, "%lld\t\t%d\t%f\n", 
				data->time,
				data->max_chromon_size,
				data->average_chromon_size);

		/* Averages */ 
		average_max_chromon += data->max_chromon_size;
		average_average_chromon += data->average_chromon_size;

		/* Max */
		max_max_chromon = 
			imax (max_max_chromon,  data->max_chromon_size);
		max_average_chromon =
			f_max (max_average_chromon, data->average_chromon_size);

		/* Time */
		average_time += data->time;
		max_time = imax (max_time, data->time);

		/* Standard Deviation */
		sd_max_chromon += isquare ( data->max_chromon_size );
		sd_av_chromon += fsquare ( data->average_chromon_size);
		sd_time += isquare ( data->time);

		/* We take care of the allocation */
		free (data);
	}

	/* We need to normalise all our averages */

	
	fprintf (ostream, "\nSummary:\n");
	
	fprintf (ostream, "Av. Max. Chr.\t= %f\n",
			((float) (((double) average_max_chromon) / ((double) i))));
	fprintf (ostream, "Av. Av. Chr.\t= %f\n",
			average_average_chromon / ((float) i));
	fprintf (ostream, "Max. Max. Chr.\t= %ld\n",
			max_max_chromon);
	fprintf (ostream, "Max. Av. Chr.\t= %f\n",
			max_average_chromon);
	fprintf (ostream, "Av. Time (us)\t= %f\n",
			((float) (((double) average_time) / ((double) i))));
	fprintf (ostream, "Max. Time (us)\t= %ld\n",
			max_time);
	fprintf (ostream, "StD. Av. Chr.\t= %f\n",
			( (float) (sqrt ( ((double) sd_av_chromon) - 
					  ((double) average_average_chromon))) / ((double) i)) );
	fprintf (ostream, "StD. Max. Chr.\t= %f\n", 
			( (float) (sqrt ( ((double) sd_max_chromon) - 
					  ((double) average_max_chromon))) / ((double) i)) );
	fprintf (ostream, "StD. Time\t= %f\n",
			( (float) (sqrt ( ((double) sd_time) - 
					  ((double) average_time))) / ((double) i)) );
	
	
			

#if 0 
	fprintf (ostream, "%f\n%f\n%ld\n%f\n%f\n%ld\n",
			((float) (((double) average_max_chromon) / ((double) i))),
			average_average_chromon / ((float) i),
			max_max_chromon,
			max_average_chromon,
			((float) (((double) average_time) / ((double) i))),
			max_time);

	fprintf (ostream, ";\n\n");
#endif


	return;
}

__inline__ static unsigned long int
imax (unsigned long int	a, unsigned long int b)
{
	return (a > b) ? a : b;
}

__inline__ static float
f_max (float a, float b)
{
	return (a > b) ? a : b;
}

__inline__ static unsigned long int
isquare (unsigned long int a)
{
	return a * a;
}

__inline__ static float
fsquare (float a)
{
	return a * a;
}

#define GET_TEXT2(REL) #REL
#define GET_TEXT(REL) GET_TEXT2(REL)

static void
help (void)
{
	version ();
	
	fprintf (stderr, "\nUsage: process options\n");
	fprintf (stderr, "\t--version, -v\t\tPrint Version and Exit\n");
	fprintf (stderr, "\t--help, -h\t\tPrint this screen\n");
	fprintf (stderr, "\t--input, -i filename\tRead graph from file filename\n");
	fprintf (stderr, "\t--output, -o filename\tOutput statistics to file filename\n");
	fprintf (stderr, "\t--max-limit, -m limit\tGenerate/Read a maximum of limit graphs\n");
	fprintf (stderr, "\t--size, -s graph_size\tRandomly generate graphs of graph_size vertices\n\n");
	fprintf (stderr, "Default behaviour:     Ask user for input for size to produce of random graph(s).\n");
	fprintf (stderr, "                       Assumes max-limit = 1, ie does a single run.\n");
	fprintf (stderr, "                       Write statistics to standard output.\n\n");
	fprintf (stderr, "Precedence behaviour:  Specifing --input or -i will override specifing --size or -s.\n");
	fprintf (stderr, "                       That is, input from file will be preferred over producing random graphs.\n");
	exit (1);
}


static void
version (void)
{
	fprintf (stderr, "Process version %s.\n", GET_TEXT(VERSION));
	fprintf (stderr, "Copyright/Copyleft (c) 2004.\nMenaka Lashitha Bandara <lashi@optusnet.com.au>.\n");	
	fprintf (stderr, "\n%s\n\n", license);
	fprintf (stderr, "Compile info:\n");
	fprintf (stderr, "\tCompiler: %s, Optimisations: %s\n", 
			GET_TEXT (CCINFO), GET_TEXT (OPTS) );
	fprintf (stderr, "\tImplimented algorithms EF, HST, EHST\n");
}

static void
clean_up (void)
{
	/* Clean up our output streams */
	if ( stream )
		fclose (stream);

	if ( stream )
		fclose (ostream);

	return;
}

#ifdef __DEBUG__
static void
colour_graph (MonoGraph *g)
{
	Colour	c [] = {BLACK, BLACK, BLACK,
		BLACK, BLACK, BLACK, BLACK,
		WHITE, BLACK, BLACK, WHITE, BLACK,
		WHITE, WHITE, WHITE, WHITE,
		WHITE, WHITE, WHITE};

	unsigned int	i;

	for ( i = 0; i < 19; i++ )
		g->vlist [i]->colour = c [i];

	return;
}
#endif /* __DEBUG__ */
