/* 
    Copyright (c) 2004, Menaka Lashitha Bandara <lashi@optusnet.com.au>

    readgraph.c - part of process.
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include <stdio.h>
#include <stdlib.h>

#include <monograph/graph.h>
#include <exception/exception.h>
#include <bool.h>

static unsigned int discover_size (FILE *instream);
static bool get_next_edge (FILE *instream, unsigned int *u, unsigned int *v);
static void eat_up_whitespace (FILE *stream);
static void put_edges (MonoGraph *g, FILE *instream);

extern unsigned int
count_graphs (FILE *instream)
{
}

extern MonoGraph 
*read_graph (FILE *instream)
{
	fpos_t		start;
	int		c;
	unsigned int	size;
	MonoGraph	*g;
	
	DEBUG_ASSERT ( instream != NULL );

	/* We get our current position */
	fgetpos (instream, &start);

	/* We need not check for beginning of graph,
	 * because discover_size () should handle it */
	size = discover_size (instream);

	if ( !size )
		return NULL;

	/* Return back to the start */
	fsetpos (instream, &start);

	g = create_graph (size);

	put_edges (g, instream);

	/* We leave our stream in a sane state */
	c = getc (instream);
	/* We need to have this */
	if ( c != ';' ) {
		exception_raise (FATAL,
				"SyntaxError",
				NONE,
				"Expecting end of graph marker ;");
		exception_handle ();
	}

	eat_up_whitespace (instream);
	
	/* Now the stream should either point to 
	 * another graph or EOF */

	return g;
}

static unsigned int
discover_size (FILE *instream)
{
	/* Edge pair */ 
	unsigned int	u, v;
	unsigned int	l = 0;

	while ( get_next_edge (instream, &u, &v) ) {
	
		/* Maintain l as max */
		if ( l < u )
			l = u;
		if ( l < v )
			l = v;
	}

#ifdef __DEBUG__
	printf ("%d\n", l+1);
#endif /* __DEBUG__ */
		
	return l + 1;
}

static bool
get_next_edge (FILE *instream, unsigned int *u, unsigned int *v)
{
	unsigned int	c;
	unsigned int	i;
	char		num [100];

	eat_up_whitespace (instream);
	c = getc (instream);

	/* We've read to the end of our graph or EOF */
	if ( c == ';' ) {
		ungetc (c, instream);
		return false;
	}

	/* Expecting tuple */
	if ( c != '(') {
		exception_raise (FATAL,
				"Syntax Error",
				NONE,
				"Expecting (");
		exception_handle ();
	}

	eat_up_whitespace (instream);
	i = 0;
	c = getc (instream);

	/* Read in our first number */
	while ( c >= '0' && c <= '9' ) {
		num [i++] = c;
		c = getc (instream);
	}
	
	ungetc (c, instream);
	
	num [i] = '\0';
	*u = atoi (num);
	

	eat_up_whitespace (instream);
	c = getc (instream);

	/* Need the comma separator between numbers */
	if ( c != ',' ) {
		exception_raise (FATAL,
				"Syntax Error",
				NONE,
				"Expecting ,");
		exception_handle ();
	}


	eat_up_whitespace (instream);
	i = 0;
	c = getc (instream);
	
	while ( c >= '0' && c <= '9' ) {
		num [i++] = c;
		c = getc (instream);
	}
	ungetc (c, instream);

	num [i] = '\0';
	*v = atoi (num);

	eat_up_whitespace (instream);
	c = getc (instream);

	if ( c != ')' ) { 
		exception_raise (FATAL,
				"Syntax Error",
				NONE,
				"Expecting )");
		exception_handle ();
	}

	/* THe only thing that can happen next is
	 * whitespace */
	eat_up_whitespace (instream);

	return true;
}

static void
eat_up_whitespace (FILE *stream)
{
	int	c;

	do {
		c = getc (stream);
	} while ( c == '\n' || c == '\t' || c == ' ' );

	ungetc (c, stream);

	return;
}

static void
put_edges (MonoGraph *g, FILE *instream)
{
	unsigned int	u, v;

	while ( get_next_edge (instream, &u, &v) ) {
		g->method->uconnect (g, u, v); 

#ifdef __DEBUG__
		printf ("edge (%d, %d)\n", u, v);
#endif /* __DEBUG__ */ 
	}

	return;
}
