#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>

#include <monograph/graph.h>
#include <generator/generator.h>
#include <algorithm/ef.h>


static void run_ef (MonoGraph *g);
static void run_hst (MonoGraph *g);
static void chromonstats (ChromonList *c);
static void check_sanity (MonoGraph *g1, MonoGraph *g2);

int
main (int argc, char **argv)
{
	MonoGraph	*g, *g_hst;
	int		i, j, max;
	int		seed;
	
	if ( argc == 1 ) { 
		printf ("Args!!!");
		return 1;
	}

	i = atoi (argv [1]);

	if ( i < 4 ) {
		printf ("Need arg > 4");
		return 1;
	}
	if ( argc > 2 )
		max = atoi (argv [2]);
	else 
		max = 1;

	if ( !max ) 
		max = 1;

	for ( j = 0; j < max; j++ ) { 
		if ( argc > 3 ) 
			g = generate_graph_seed (i, atoi (argv [3]));
		else 
			g = generate_graph (i);
	
	
		/* Ok, we duplicate the graph */
		g_hst = g->method->duplicate (g);
	
		check_sanity (g, g_hst);
	
		run_ef (g);
		run_hst (g_hst);
	}

	return 0;
}

static void
run_ef (MonoGraph *g)
{
	ChromonList	*c;
	struct rusage	myuse [2];
	
	printf ("***********************\n");
	printf ("Running EF algorithm...\n");
	printf ("***********************\n");
	
	getrusage (RUSAGE_SELF, myuse);
	/* Ok, we do an EF */
	c = ef_algorithm (g);
	getrusage (RUSAGE_SELF, myuse + 1);

	/* This is the times used */
	printf ("Times:\n");
	printf ("\tSystem\t\t= %ds %dus\n", 
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec);
	printf ("\tUser\t\t= %ds %dus\n",
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
	printf ("\tTotal\t\t= %ds %dus\n",
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec +
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec +  
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);

	chromonstats (c); 

	delete_chromonlist (c);
	delete_graph (g);

	return;
}


static void
run_hst (MonoGraph *g)
{
	ChromonList	*c;
	struct rusage	myuse [2];
	
	printf ("************************\n");
	printf ("Running HST algorithm...\n");
	printf ("************************\n");

	
	getrusage (RUSAGE_SELF, myuse);
	/* Ok, we do a HST */
	c = hst_algorithm (g);
	getrusage (RUSAGE_SELF, myuse + 1);

	/* This is the times used */
	printf ("Times:\n");
	printf ("\tSystem\t\t= %ds %dus\n", 
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec);
	printf ("\tUser\t\t= %ds %dus\n",
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
	printf ("\tTotal\t\t= %ds %dus\n",
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec +
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec +  
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);

	chromonstats (c);

	delete_chromonlist (c);
	delete_graph (g);
	
	return; 
}

static void
chromonstats (ChromonList *c)
{
	unsigned int	i;
	unsigned int 	largest;
	List		*l;
	
	printf ("Average Chromon Size = %f\n", 
			((float) c->graph->vertices) / ((float) (c->black->length + c->white->length)));

	c->black->method->reset (c->black);
	c->white->method->reset (c->white);
	
	/* Hunt for the largest chromon */
	for ( i = 0, largest = 0; i < c->black->length; i++ )
		if ( (l = c->black->method->next (c->black) ))
			if ( l->length > largest )
				largest = l->length;

	for ( i = 0; i < c->white->length; i++ )
		if ( (l = c->white->method->next (c->white) ))
			if ( l->length > largest )
				largest = l->length;

	printf ("Max. Chromon Size = %d\n", largest);

	return;
}

static void
check_sanity (MonoGraph *g1, MonoGraph *g2)
{
	unsigned int	i, j;
	
	for (i = 0; i < g1->vertices; i++ ) {
		if ( g1->vlist [i]->colour !=
				g2->vlist [i]->colour )
			printf ("Colour mismatch, vertex %d\n", i);

		for ( j = 0; j < g1->vlist [i]->neighbours;  j++ ) {
			if ( g1->vlist [i]->neighbour [j]->id  !=
					g2->vlist [i]->neighbour [j]->id )
				printf ("Neighbour mismatch, vertex %d\n", i);
		}
	}
	

	return;
}
