#include <stdio.h>
#include <stdlib.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <unistd.h>

#include <monograph/graph.h>
#include <generator/generator.h>
#include <algorithm/ef.h>
#include <algorithm/hst.h>

#define MAX 500

static void run (MonoGraph *g, MonoGraph *g_hst);
static void run_ef (MonoGraph *g);
static void run_hst (MonoGraph *g);
static void run_qhst (MonoGraph *g);
static void chromonstats (ChromonList *c, unsigned int what);
static void check_sanity (MonoGraph *g1, MonoGraph *g2);
static void compare_vals (void);

static unsigned int qhst_lchromon;
static float qhst_achromon;
static unsigned int hst_lchromon;
static float hst_achromon;


int
main (int argc, char **argv)
{
	MonoGraph	*g, *g_hst;
	int		i;
	unsigned int	j;
	int		seed;
	
	if ( argc == 1 ) { 
		printf ("Args!!!");
		return 1;
	}

	i = atoi (argv [1]);

	if ( i < 4 ) {
		printf ("Need arg > 4");
		return 1;
	}
	for ( j = 0; j < MAX; j++ ) {

		if ( argc == 3 ) 
			g = generate_graph_seed (i, atoi (argv [2]));
		else 
			g = generate_graph (i);
		/* Ok, we duplicate the graph */
		g_hst = g->method->duplicate (g);

		check_sanity (g, g_hst);

		run (g, g_hst);
	}

	return 0;
}

static void
run (MonoGraph *g, MonoGraph *g_hst)
{
	run_qhst (g);
	run_hst (g_hst);

	compare_vals ();

	return;
}


static void
run_hst (MonoGraph *g)
{
	ChromonList	*c;
	struct rusage	myuse [2];
	
	getrusage (RUSAGE_SELF, myuse);
	/* Ok, we do a HST */
	c = hst_algorithm (g);
	getrusage (RUSAGE_SELF, myuse + 1);

#if 0
	/* This is the times used */
	printf ("Times:\n");
	printf ("\tSystem\t\t= %ds %dus\n", 
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec);
	printf ("\tUser\t\t= %ds %dus\n",
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
	printf ("\tTotal\t\t= %ds %dus\n",
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec +
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec +  
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
#endif

	chromonstats (c,0);

	delete_chromonlist (c);
	delete_graph (g);
	
	return; 
}

static void
run_qhst (MonoGraph *g)
{
	ChromonList	*c;
	struct rusage	myuse [2];
	
	getrusage (RUSAGE_SELF, myuse);
	/* Ok, we do a HST */
	c = hst_algorithm (g);
	getrusage (RUSAGE_SELF, myuse + 1);

#if 0 
	/* This is the times used */
	printf ("Times:\n");
	printf ("\tSystem\t\t= %ds %dus\n", 
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec);
	printf ("\tUser\t\t= %ds %dus\n",
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
	printf ("\tTotal\t\t= %ds %dus\n",
			myuse [1].ru_stime.tv_sec - myuse [0].ru_stime.tv_sec +
			myuse [1].ru_utime.tv_sec - myuse [0].ru_utime.tv_sec,
			myuse [1].ru_stime.tv_usec - myuse [0].ru_stime.tv_usec +  
			myuse [1].ru_utime.tv_usec - myuse [0].ru_utime.tv_usec);
#endif

	chromonstats (c,1);

	delete_chromonlist (c);
	delete_graph (g);
	
	return; 
}

static void
chromonstats (ChromonList *c, unsigned int what)
{
	unsigned int	i;
	unsigned int 	largest;
	List		*l;
	
	switch (what) {
		case 0: hst_achromon = ((float) c->graph->vertices) / ((float) (c->black->length + c->white->length));
			break; 
		case 1: qhst_achromon = ((float) c->graph->vertices) / ((float) (c->black->length + c->white->length));
			break;
	}
	

	c->black->method->reset (c->black);
	c->white->method->reset (c->white);
	
	/* Hunt for the largest chromon */
	for ( i = 0, largest = 0; i < c->black->length; i++ )
		if ( (l = c->black->method->next (c->black) ))
			if ( l->length > largest )
				largest = l->length;

	for ( i = 0; i < c->white->length; i++ )
		if ( (l = c->white->method->next (c->white) ))
			if ( l->length > largest )
				largest = l->length;

	switch (what) {
		case 0: hst_lchromon = largest; break;
		case 1: qhst_lchromon = largest; break;
	}

	return;
}

static void
check_sanity (MonoGraph *g1, MonoGraph *g2)
{
	unsigned int	i, j;
	
	for (i = 0; i < g1->vertices; i++ ) {
		if ( g1->vlist [i]->colour !=
				g2->vlist [i]->colour )
			printf ("Colour mismatch, vertex %d\n", i);

		for ( j = 0; j < g1->vlist [i]->neighbours;  j++ ) {
			if ( g1->vlist [i]->neighbour [j]->id  !=
					g2->vlist [i]->neighbour [j]->id )
				printf ("Neighbour mismatch, vertex %d\n", i);
		}
	}
	

	return;
}

static void
compare_vals (void)
{
	if ( (hst_lchromon == qhst_lchromon)  &&
			(hst_achromon == qhst_achromon) )
		return;

	/* Oh oh... We've disproved! */
	printf ("AHHHHH!!!\n");
	printf ("HST(lchromon, achromon) = (%d, %d)",
			hst_lchromon, hst_achromon);
	printf ("QHST(lchromon, achromon) = (%d, %d)",
			qhst_lchromon, qhst_achromon);

	return;
}
